/*
 * Decompiled with CFR 0.152.
 */
package io.prestosql.server.security.jwt;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import io.airlift.json.JsonCodec;
import io.airlift.log.Logger;
import io.prestosql.server.security.jwt.EcCurve;
import java.math.BigInteger;
import java.security.PublicKey;
import java.security.interfaces.ECPublicKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.ECParameterSpec;
import java.security.spec.ECPoint;
import java.util.Base64;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;

public final class JwkDecoder {
    private static final Logger log = Logger.get(JwkDecoder.class);
    private static final JsonCodec<Keys> KEYS_CODEC = JsonCodec.jsonCodec(Keys.class);

    private JwkDecoder() {
    }

    public static Map<String, PublicKey> decodeKeys(String jwkJson) {
        Keys keys = (Keys)KEYS_CODEC.fromJson(jwkJson);
        return (Map)keys.getKeys().stream().map(JwkDecoder::tryDecodeJwkKey).filter(Optional::isPresent).map(Optional::get).collect(ImmutableMap.toImmutableMap(JwkPublicKey::getKeyId, Function.identity()));
    }

    public static Optional<? extends JwkPublicKey> tryDecodeJwkKey(Map<String, String> properties) {
        String keyType;
        String keyId = properties.get("kid");
        if (Strings.isNullOrEmpty((String)keyId)) {
            return Optional.empty();
        }
        switch (keyType = properties.get("kty")) {
            case "RSA": {
                return JwkDecoder.tryDecodeRsaKey(keyId, properties);
            }
            case "EC": {
                return JwkDecoder.tryDecodeEcKey(keyId, properties);
            }
        }
        return Optional.empty();
    }

    public static Optional<JwkRsaPublicKey> tryDecodeRsaKey(String keyId, Map<String, String> properties) {
        String encodedModulus = properties.get("n");
        if (Strings.isNullOrEmpty((String)encodedModulus)) {
            log.error("JWK RSA key %s does not contain the required modulus field 'n'", new Object[]{keyId});
            return Optional.empty();
        }
        String encodedExponent = properties.get("e");
        if (Strings.isNullOrEmpty((String)encodedExponent)) {
            log.error("JWK RSA key %s does not contain the required exponent field 'e'", new Object[]{keyId});
            return Optional.empty();
        }
        Optional<BigInteger> modulus = JwkDecoder.decodeBigint(keyId, "modulus", encodedModulus);
        if (modulus.isEmpty()) {
            return Optional.empty();
        }
        Optional<BigInteger> exponent = JwkDecoder.decodeBigint(keyId, "exponent", encodedExponent);
        if (exponent.isEmpty()) {
            return Optional.empty();
        }
        return Optional.of(new JwkRsaPublicKey(keyId, exponent.get(), modulus.get()));
    }

    public static Optional<JwkEcPublicKey> tryDecodeEcKey(String keyId, Map<String, String> properties) {
        String curveName = properties.get("crv");
        if (Strings.isNullOrEmpty((String)curveName)) {
            log.error("JWK EC key %s does not contain the required curve field 'crv'", new Object[]{keyId});
            return Optional.empty();
        }
        String encodedX = properties.get("x");
        if (Strings.isNullOrEmpty((String)encodedX)) {
            log.error("JWK EC key %s does not contain the required x coordinate field 'x'", new Object[]{keyId});
            return Optional.empty();
        }
        String encodedY = properties.get("y");
        if (Strings.isNullOrEmpty((String)encodedY)) {
            log.error("JWK EC key %s does not contain the required y coordinate field 'y'", new Object[]{keyId});
            return Optional.empty();
        }
        Optional<ECParameterSpec> curve = EcCurve.tryGet(curveName);
        if (curve.isEmpty()) {
            log.error("JWK EC %s curve '%s' is not supported", new Object[]{keyId, curveName});
            return Optional.empty();
        }
        Optional<BigInteger> x = JwkDecoder.decodeBigint(keyId, "x", encodedX);
        if (x.isEmpty()) {
            return Optional.empty();
        }
        Optional<BigInteger> y = JwkDecoder.decodeBigint(keyId, "y", encodedY);
        if (y.isEmpty()) {
            return Optional.empty();
        }
        ECPoint w = new ECPoint(x.get(), y.get());
        return Optional.of(new JwkEcPublicKey(keyId, curve.get(), w));
    }

    private static Optional<BigInteger> decodeBigint(String keyId, String fieldName, String encodedNumber) {
        try {
            return Optional.of(new BigInteger(1, Base64.getUrlDecoder().decode(encodedNumber)));
        }
        catch (IllegalArgumentException e) {
            log.error((Throwable)e, "JWK %s %s is not a valid number", new Object[]{keyId, fieldName});
            return Optional.empty();
        }
    }

    public static class Keys {
        private final List<Map<String, String>> keys;

        @JsonCreator
        public Keys(@JsonProperty(value="keys") List<Map<String, String>> keys) {
            this.keys = ImmutableList.copyOf((Collection)Objects.requireNonNull(keys, "keys is null"));
        }

        public List<Map<String, String>> getKeys() {
            return this.keys;
        }
    }

    public static class JwkEcPublicKey
    implements JwkPublicKey,
    ECPublicKey {
        private final String keyId;
        private final ECParameterSpec parameterSpec;
        private final ECPoint w;

        public JwkEcPublicKey(String keyId, ECParameterSpec parameterSpec, ECPoint w) {
            this.keyId = Objects.requireNonNull(keyId, "keyId is null");
            this.parameterSpec = Objects.requireNonNull(parameterSpec, "parameterSpec is null");
            this.w = Objects.requireNonNull(w, "w is null");
        }

        @Override
        public String getKeyId() {
            return this.keyId;
        }

        @Override
        public ECParameterSpec getParams() {
            return this.parameterSpec;
        }

        @Override
        public ECPoint getW() {
            return this.w;
        }

        @Override
        public String getAlgorithm() {
            return "EC";
        }

        @Override
        public String getFormat() {
            return "JWK";
        }

        @Override
        public byte[] getEncoded() {
            throw new UnsupportedOperationException();
        }
    }

    public static class JwkRsaPublicKey
    implements JwkPublicKey,
    RSAPublicKey {
        private final String keyId;
        private final BigInteger modulus;
        private final BigInteger exponent;

        public JwkRsaPublicKey(String keyId, BigInteger exponent, BigInteger modulus) {
            this.keyId = Objects.requireNonNull(keyId, "keyId is null");
            this.exponent = Objects.requireNonNull(exponent, "exponent is null");
            this.modulus = Objects.requireNonNull(modulus, "modulus is null");
        }

        @Override
        public String getKeyId() {
            return this.keyId;
        }

        @Override
        public BigInteger getModulus() {
            return this.modulus;
        }

        @Override
        public BigInteger getPublicExponent() {
            return this.exponent;
        }

        @Override
        public String getAlgorithm() {
            return "RSA";
        }

        @Override
        public String getFormat() {
            return "JWK";
        }

        @Override
        public byte[] getEncoded() {
            throw new UnsupportedOperationException();
        }
    }

    public static interface JwkPublicKey
    extends PublicKey {
        public String getKeyId();
    }
}

