/*
 * Decompiled with CFR 0.152.
 */
package io.prestosql.spiller;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.AbstractIterator;
import com.google.common.collect.ImmutableList;
import com.google.common.io.Closer;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import io.airlift.slice.InputStreamSliceInput;
import io.airlift.slice.OutputStreamSliceOutput;
import io.airlift.slice.SliceInput;
import io.airlift.slice.SliceOutput;
import io.prestosql.execution.buffer.PagesSerde;
import io.prestosql.execution.buffer.PagesSerdeUtil;
import io.prestosql.execution.buffer.SerializedPage;
import io.prestosql.memory.context.LocalMemoryContext;
import io.prestosql.operator.SpillContext;
import io.prestosql.spi.ErrorCodeSupplier;
import io.prestosql.spi.Page;
import io.prestosql.spi.PrestoException;
import io.prestosql.spi.StandardErrorCode;
import io.prestosql.spiller.FileHolder;
import io.prestosql.spiller.SingleStreamSpiller;
import io.prestosql.spiller.SpillCipher;
import io.prestosql.spiller.SpillerStats;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import javax.annotation.concurrent.NotThreadSafe;

@NotThreadSafe
public class FileSingleStreamSpiller
implements SingleStreamSpiller {
    @VisibleForTesting
    static final int BUFFER_SIZE = 4096;
    private final FileHolder targetFile;
    private final Closer closer = Closer.create();
    private final PagesSerde serde;
    private final SpillerStats spillerStats;
    private final SpillContext localSpillContext;
    private final LocalMemoryContext memoryContext;
    private final ListeningExecutorService executor;
    private boolean writable = true;
    private long spilledPagesInMemorySize;
    private ListenableFuture<?> spillInProgress = Futures.immediateFuture(null);
    private final Runnable fileSystemErrorHandler;

    public FileSingleStreamSpiller(PagesSerde serde, ListeningExecutorService executor, Path spillPath, SpillerStats spillerStats, SpillContext spillContext, LocalMemoryContext memoryContext, Optional<SpillCipher> spillCipher, Runnable fileSystemErrorHandler) {
        this.serde = Objects.requireNonNull(serde, "serde is null");
        this.executor = Objects.requireNonNull(executor, "executor is null");
        this.spillerStats = Objects.requireNonNull(spillerStats, "spillerStats is null");
        this.localSpillContext = spillContext.newLocalSpillContext();
        this.memoryContext = Objects.requireNonNull(memoryContext, "memoryContext is null");
        if (Objects.requireNonNull(spillCipher, "spillCipher is null").isPresent()) {
            this.closer.register(spillCipher.get()::close);
        }
        this.memoryContext.setBytes(4096L);
        this.fileSystemErrorHandler = Objects.requireNonNull(fileSystemErrorHandler, "filesystemErrorHandler is null");
        try {
            this.targetFile = (FileHolder)this.closer.register((Closeable)new FileHolder(Files.createTempFile(spillPath, "spill", ".bin", new FileAttribute[0])));
        }
        catch (IOException e) {
            this.fileSystemErrorHandler.run();
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_INTERNAL_ERROR, "Failed to create spill file", (Throwable)e);
        }
    }

    @Override
    public ListenableFuture<?> spill(Iterator<Page> pageIterator) {
        Objects.requireNonNull(pageIterator, "pageIterator is null");
        this.checkNoSpillInProgress();
        this.spillInProgress = this.executor.submit(() -> this.writePages(pageIterator));
        return this.spillInProgress;
    }

    @Override
    public long getSpilledPagesInMemorySize() {
        return this.spilledPagesInMemorySize;
    }

    @Override
    public Iterator<Page> getSpilledPages() {
        this.checkNoSpillInProgress();
        return this.readPages();
    }

    @Override
    public ListenableFuture<List<Page>> getAllSpilledPages() {
        return this.executor.submit(() -> ImmutableList.copyOf(this.getSpilledPages()));
    }

    private void writePages(Iterator<Page> pageIterator) {
        Preconditions.checkState((boolean)this.writable, (Object)"Spilling no longer allowed. The spiller has been made non-writable on first read for subsequent reads to be consistent");
        try (OutputStreamSliceOutput output = new OutputStreamSliceOutput(this.targetFile.newOutputStream(StandardOpenOption.APPEND), 4096);
             PagesSerde.PagesSerdeContext context = this.serde.newContext();){
            while (pageIterator.hasNext()) {
                Page page = pageIterator.next();
                this.spilledPagesInMemorySize += page.getSizeInBytes();
                SerializedPage serializedPage = this.serde.serialize(context, page);
                long pageSize = serializedPage.getSizeInBytes();
                this.localSpillContext.updateBytes(pageSize);
                this.spillerStats.addToTotalSpilledBytes(pageSize);
                PagesSerdeUtil.writeSerializedPage((SliceOutput)output, serializedPage);
            }
        }
        catch (IOException | UncheckedIOException e) {
            this.fileSystemErrorHandler.run();
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_INTERNAL_ERROR, "Failed to spill pages", (Throwable)e);
        }
    }

    private Iterator<Page> readPages() {
        Preconditions.checkState((boolean)this.writable, (Object)"Repeated reads are disallowed to prevent potential resource leaks");
        this.writable = false;
        try {
            InputStream input = (InputStream)this.closer.register((Closeable)this.targetFile.newInputStream(new OpenOption[0]));
            Iterator<Page> pages = PagesSerdeUtil.readPages(this.serde, (SliceInput)new InputStreamSliceInput(input, 4096));
            return FileSingleStreamSpiller.closeWhenExhausted(pages, input);
        }
        catch (IOException e) {
            this.fileSystemErrorHandler.run();
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_INTERNAL_ERROR, "Failed to read spilled pages", (Throwable)e);
        }
    }

    @Override
    public void close() {
        this.closer.register((Closeable)this.localSpillContext);
        this.closer.register(() -> this.memoryContext.setBytes(0L));
        try {
            this.closer.close();
        }
        catch (IOException e) {
            this.fileSystemErrorHandler.run();
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_INTERNAL_ERROR, "Failed to close spiller", (Throwable)e);
        }
    }

    private void checkNoSpillInProgress() {
        Preconditions.checkState((boolean)this.spillInProgress.isDone(), (Object)"spill in progress");
    }

    private static <T> Iterator<T> closeWhenExhausted(final Iterator<T> iterator, final Closeable resource) {
        Objects.requireNonNull(iterator, "iterator is null");
        Objects.requireNonNull(resource, "resource is null");
        return new AbstractIterator<T>(){

            protected T computeNext() {
                if (iterator.hasNext()) {
                    return iterator.next();
                }
                try {
                    resource.close();
                }
                catch (IOException e) {
                    throw new UncheckedIOException(e);
                }
                return this.endOfData();
            }
        };
    }
}

