/*
 * Decompiled with CFR 0.152.
 */
package io.prestosql.spiller;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.ImmutableList;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.inject.Inject;
import io.airlift.concurrent.Threads;
import io.airlift.log.Logger;
import io.prestosql.execution.buffer.PagesSerde;
import io.prestosql.execution.buffer.PagesSerdeFactory;
import io.prestosql.memory.context.LocalMemoryContext;
import io.prestosql.metadata.Metadata;
import io.prestosql.operator.SpillContext;
import io.prestosql.spi.ErrorCodeSupplier;
import io.prestosql.spi.PrestoException;
import io.prestosql.spi.StandardErrorCode;
import io.prestosql.spi.block.BlockEncodingSerde;
import io.prestosql.spi.type.Type;
import io.prestosql.spiller.AesSpillCipher;
import io.prestosql.spiller.FileSingleStreamSpiller;
import io.prestosql.spiller.NodeSpillConfig;
import io.prestosql.spiller.SingleStreamSpiller;
import io.prestosql.spiller.SingleStreamSpillerFactory;
import io.prestosql.spiller.SpillCipher;
import io.prestosql.spiller.SpillerStats;
import io.prestosql.sql.analyzer.FeaturesConfig;
import java.io.IOException;
import java.nio.file.DirectoryStream;
import java.nio.file.FileStore;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.time.Duration;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

public class FileSingleStreamSpillerFactory
implements SingleStreamSpillerFactory {
    private static final Logger log = Logger.get(FileSingleStreamSpillerFactory.class);
    @VisibleForTesting
    static final String SPILL_FILE_PREFIX = "spill";
    @VisibleForTesting
    static final String SPILL_FILE_SUFFIX = ".bin";
    private static final String SPILL_FILE_GLOB = "spill*.bin";
    private static final Duration SPILL_PATH_HEALTH_EXPIRY_INTERVAL = Duration.ofMinutes(5L);
    private final ListeningExecutorService executor;
    private final PagesSerdeFactory serdeFactory;
    private final List<Path> spillPaths;
    private final SpillerStats spillerStats;
    private final double maxUsedSpaceThreshold;
    private final boolean spillEncryptionEnabled;
    private int roundRobinIndex;
    private final LoadingCache<Path, Boolean> spillPathHealthCache;

    @Inject
    public FileSingleStreamSpillerFactory(Metadata metadata, SpillerStats spillerStats, FeaturesConfig featuresConfig, NodeSpillConfig nodeSpillConfig) {
        this(MoreExecutors.listeningDecorator((ExecutorService)Executors.newFixedThreadPool(Objects.requireNonNull(featuresConfig, "featuresConfig is null").getSpillerThreads(), Threads.daemonThreadsNamed((String)"binary-spiller-%s"))), Objects.requireNonNull(metadata, "metadata is null").getBlockEncodingSerde(), spillerStats, Objects.requireNonNull(featuresConfig, "featuresConfig is null").getSpillerSpillPaths(), Objects.requireNonNull(featuresConfig, "featuresConfig is null").getSpillMaxUsedSpaceThreshold(), Objects.requireNonNull(nodeSpillConfig, "nodeSpillConfig is null").isSpillCompressionEnabled(), Objects.requireNonNull(nodeSpillConfig, "nodeSpillConfig is null").isSpillEncryptionEnabled());
    }

    @VisibleForTesting
    public FileSingleStreamSpillerFactory(ListeningExecutorService executor, BlockEncodingSerde blockEncodingSerde, SpillerStats spillerStats, List<Path> spillPaths, double maxUsedSpaceThreshold, boolean spillCompressionEnabled, boolean spillEncryptionEnabled) {
        this.serdeFactory = new PagesSerdeFactory(blockEncodingSerde, spillCompressionEnabled);
        this.executor = Objects.requireNonNull(executor, "executor is null");
        this.spillerStats = Objects.requireNonNull(spillerStats, "spillerStats cannot be null");
        Objects.requireNonNull(spillPaths, "spillPaths is null");
        this.spillPaths = ImmutableList.copyOf(spillPaths);
        spillPaths.forEach(path -> {
            try {
                Files.createDirectories(path, new FileAttribute[0]);
            }
            catch (IOException e) {
                throw new IllegalArgumentException(String.format("could not create spill path %s; adjust %s config property or filesystem permissions", path, "spiller-spill-path"), e);
            }
            if (!this.isAccessible((Path)path)) {
                throw new IllegalArgumentException(String.format("spill path %s is not accessible, it must be +rwx; adjust %s config property or filesystem permissions", path, "spiller-spill-path"));
            }
        });
        this.maxUsedSpaceThreshold = maxUsedSpaceThreshold;
        this.spillEncryptionEnabled = spillEncryptionEnabled;
        this.roundRobinIndex = 0;
        this.spillPathHealthCache = CacheBuilder.newBuilder().expireAfterWrite(SPILL_PATH_HEALTH_EXPIRY_INTERVAL).build(CacheLoader.from(path -> this.isAccessible((Path)path) && this.isSeeminglyHealthy((Path)path)));
    }

    @PostConstruct
    public void cleanupOldSpillFiles() {
        this.spillPaths.forEach(FileSingleStreamSpillerFactory::cleanupOldSpillFiles);
    }

    @PreDestroy
    public void destroy() {
        this.executor.shutdownNow();
    }

    private static void cleanupOldSpillFiles(Path path) {
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(path, SPILL_FILE_GLOB);){
            stream.forEach(spillFile -> {
                try {
                    log.info("Deleting old spill file: " + spillFile);
                    Files.delete(spillFile);
                }
                catch (Exception e) {
                    log.warn("Could not cleanup old spill file: " + spillFile);
                }
            });
        }
        catch (IOException e) {
            log.warn((Throwable)e, "Error cleaning spill files");
        }
    }

    @Override
    public SingleStreamSpiller create(List<Type> types, SpillContext spillContext, LocalMemoryContext memoryContext) {
        Optional<SpillCipher> spillCipher = Optional.empty();
        if (this.spillEncryptionEnabled) {
            spillCipher = Optional.of(new AesSpillCipher());
        }
        PagesSerde serde = this.serdeFactory.createPagesSerdeForSpill(spillCipher);
        return new FileSingleStreamSpiller(serde, this.executor, this.getNextSpillPath(), this.spillerStats, spillContext, memoryContext, spillCipher, () -> this.spillPathHealthCache.invalidateAll());
    }

    private synchronized Path getNextSpillPath() {
        int spillPathsCount = this.spillPaths.size();
        for (int i = 0; i < spillPathsCount; ++i) {
            int pathIndex = (this.roundRobinIndex + i) % spillPathsCount;
            Path path = this.spillPaths.get(pathIndex);
            if (!this.hasEnoughDiskSpace(path) || !((Boolean)this.spillPathHealthCache.getUnchecked((Object)path)).booleanValue()) continue;
            this.roundRobinIndex = (this.roundRobinIndex + i + 1) % spillPathsCount;
            return path;
        }
        if (this.spillPaths.isEmpty()) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.OUT_OF_SPILL_SPACE, "No spill paths configured");
        }
        throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.OUT_OF_SPILL_SPACE, "No free or healthy space available for spill");
    }

    private boolean hasEnoughDiskSpace(Path path) {
        try {
            FileStore fileStore = Files.getFileStore(path);
            return (double)fileStore.getUsableSpace() > (double)fileStore.getTotalSpace() * (1.0 - this.maxUsedSpaceThreshold);
        }
        catch (IOException e) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.OUT_OF_SPILL_SPACE, "Cannot determine free space for spill", (Throwable)e);
        }
    }

    private boolean isAccessible(Path path) {
        return Files.isReadable(path) && Files.isWritable(path) && Files.isExecutable(path);
    }

    private boolean isSeeminglyHealthy(Path path) {
        try {
            Path healthTemp = Files.createTempFile(path, SPILL_FILE_PREFIX, "healthcheck", new FileAttribute[0]);
            return Files.deleteIfExists(healthTemp);
        }
        catch (IOException e) {
            log.warn((Throwable)e, "Health check failed for spill %s", new Object[]{path});
            return false;
        }
    }

    @VisibleForTesting
    long getSpillPathCacheSize() {
        return this.spillPathHealthCache.size();
    }
}

