;; Copyright (c) Cognitect, Inc.
;; All rights reserved.

(ns cognitect.s3-creds.store
  (:require
   [clojure.core.async :as a :refer (>!! thread)]
   [cognitect.s3-creds.common :refer :all]
   [cognitect.transit :as t])
  (:import
   [com.amazonaws.auth AWSCredentialsProvider]
   [com.amazonaws.auth.profile ProfileCredentialsProvider]
   [com.amazonaws.services.s3 AmazonS3ClientBuilder AmazonS3Client]
   [com.amazonaws.services.s3.model AmazonS3Exception GetObjectRequest ObjectMetadata]
   [java.io ByteArrayInputStream ByteArrayOutputStream]))

(set! *warn-on-reflection* true)

(defn- client-builder
  ([] (client-builder {}))
  ([{:keys [^AWSCredentialsProvider creds-provider
              creds-profile region]}]
   (cond-> (AmazonS3ClientBuilder/standard)
           creds-profile ^AmazonS3ClientBuilder (.withCredentials (ProfileCredentialsProvider. creds-profile))
           creds-provider ^AmazonS3ClientBuilder (.withCredentials creds-provider)
           region ^AmazonS3ClientBuilder (.withRegion ^String region)
           true .build)))

(def ^AmazonS3Client client
  (memoize client-builder))

(defn- read-transit-json
  "Reads transit from stream."
  [istm]
  (-> istm (t/reader :json) t/read))

(defn- read-s3
  [^AmazonS3Client client ^String bucket ^String key reader]
  (with-open [s3-obj (->> (GetObjectRequest. bucket key)
                          (.getObject client))
              stm (.getObjectContent s3-obj)]
    (reader stm)))

(defn- transit-json-bytes
  "Returns an array with transit json bytes for obj."
  [obj]
  (let [baos (ByteArrayOutputStream.)]
    (-> baos (t/writer :json-verbose) (t/write obj))
    (.toByteArray baos)))

(defn- bucket-and-path
  [s3-path]
  (when-let [[_ bucket base] (re-find #"s3://([^/]+)/(.+)" s3-path)]
    [bucket base]))

(defn- write-s3
  [^AmazonS3Client client ^String bucket ^String key stream length]
  (.putObject client bucket key stream (doto (ObjectMetadata.)
                                         (.setContentLength length))))

(defn- fault
  [^Throwable t]
  {:cognitect.anomalies/category :cognitect.anomalies/fault
   :cognitect.anomalies/message (.getMessage t)})

(defn get-val
  "Read the transit-encoded value in the .keys file under
s3-path."
  ([s3-path] (get-val (client) s3-path))
  ([client s3-path]
   (if-let [[bucket path] (bucket-and-path s3-path)]
     (try
       {:val (read-s3 client bucket path read-transit-json)}
       (catch AmazonS3Exception se
         (cond
           (= 301 (.getStatusCode se))
           (not-found-msg (.getMessage se))

           (= 404 (.getStatusCode se))
           not-found

           (= 403 (.getStatusCode se))
           forbidden

           :default
           (fault se)))
       (catch Throwable t
         (fault t)))
     incorrect)))

(defn put-val
  "Writes transit-encoded v in the .keys file under s3-path."
  ([s3-path v] (put-val (client) s3-path v))
  ([client s3-path v]
   (if-let [[bucket path] (bucket-and-path s3-path)]
     (try
       (let [bs (transit-json-bytes v)]
         (write-s3 client bucket path (ByteArrayInputStream. bs) (count bs)))
       {:val v}
       (catch AmazonS3Exception se
         (if (= 403 (.getStatusCode se))
           forbidden
           (fault se)))
       (catch Throwable t
         (fault t)))
     incorrect)))

(defn- result-ch
  [result]
  (doto (a/promise-chan) (>!! result)))

(defprotocol Throttle
  (-throttle?
   [_ data]
   "Boolean: should activity associated with data be throttled?"))

(defprotocol ReadStore
  (get-keyfile [_ keyfile-name] "Get keyfile from s3"))

(deftype ReadStoreImpl
  [policy client]
  ReadStore
  (get-keyfile
   [_ keyfile-name]
   (if (-throttle? policy keyfile-name)
     (result-ch busy)
     (thread
      (let [result (get-val client keyfile-name)]
        (if-let [v (:val result)]
          v
          result))))))

(deftype ThrottleImpl [last gap]
  Throttle
  (-throttle?
   [_ f]
   (let [x (System/currentTimeMillis)]
     (if (< x (+ @last gap))
       true
       (do
         (reset! last x)
         false)))))

(def never-throttle
  (reify
   Throttle
   (-throttle? [_ req] false)))

(defn create-store
  ([] (create-store {:policy never-throttle}))
  ([{:keys [policy creds-provider creds-profile]
     :or {policy never-throttle}
     :as params-or-policy}]
   (when (and creds-profile creds-provider)
     (throw (ex-info (str ":creds-profile and :creds-provider are mutually exclusive config options " params-or-policy) {:config params-or-policy})))
   (let [policy (if (map? params-or-policy)
                  policy
                  params-or-policy)
         s3 (client (select-keys params-or-policy [:creds-provider :creds-profile :region]))]
     (->ReadStoreImpl policy s3))))
