/*
 * Decompiled with CFR 0.152.
 */
package io.airlift.http.server;

import io.airlift.http.server.HttpServer;
import io.airlift.http.server.HttpServerConfig;
import io.airlift.log.Logger;
import io.airlift.security.pem.PemReader;
import java.io.File;
import java.io.IOException;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.eclipse.jetty.util.ssl.SslContextFactory;

final class ReloadableSslContextFactoryProvider {
    private static final Logger log = Logger.get(ReloadableSslContextFactoryProvider.class);
    private final SslContextFactory.Server sslContextFactory;
    private final List<String> includedCipherSuites;
    private final List<String> excludedCipherSuites;
    private final String keystorePath;
    private final String keystorePassword;
    private final String keyManagerPassword;
    private final String trustStorePath;
    private final String trustStorePassword;
    private final String secureRandomAlgorithm;
    private final int sslSessionTimeoutSeconds;
    private final int sslSessionCacheSize;
    private final HttpServer.ClientCertificate clientCertificate;

    public ReloadableSslContextFactoryProvider(HttpServerConfig config, ScheduledExecutorService scheduledExecutor, HttpServer.ClientCertificate clientCertificate) {
        Objects.requireNonNull(config, "config is null");
        Objects.requireNonNull(scheduledExecutor, "scheduledExecutor is null");
        this.includedCipherSuites = config.getHttpsIncludedCipherSuites();
        this.excludedCipherSuites = config.getHttpsExcludedCipherSuites();
        this.keystorePath = config.getKeystorePath();
        this.keystorePassword = config.getKeystorePassword();
        this.keyManagerPassword = config.getKeyManagerPassword();
        this.trustStorePath = config.getTrustStorePath();
        this.trustStorePassword = config.getTrustStorePassword();
        this.secureRandomAlgorithm = config.getSecureRandomAlgorithm();
        this.sslSessionTimeoutSeconds = Math.toIntExact(config.getSslSessionTimeout().roundTo(TimeUnit.SECONDS));
        this.sslSessionCacheSize = config.getSslSessionCacheSize();
        this.clientCertificate = Objects.requireNonNull(clientCertificate, "clientCertificate is null");
        this.sslContextFactory = this.buildContextFactory();
        long refreshTime = config.getSslContextRefreshTime().toMillis();
        scheduledExecutor.scheduleWithFixedDelay(this::reload, refreshTime, refreshTime, TimeUnit.MILLISECONDS);
    }

    private SslContextFactory.Server buildContextFactory() {
        SslContextFactory.Server sslContextFactory = new SslContextFactory.Server();
        Optional<KeyStore> pemKeyStore = ReloadableSslContextFactoryProvider.tryLoadPemKeyStore(this.keystorePath, this.keystorePassword);
        if (pemKeyStore.isPresent()) {
            sslContextFactory.setKeyStore(pemKeyStore.get());
            sslContextFactory.setKeyStorePassword("");
        } else {
            sslContextFactory.setKeyStorePath(this.keystorePath);
            sslContextFactory.setKeyStorePassword(this.keystorePassword);
            if (this.keyManagerPassword != null) {
                sslContextFactory.setKeyManagerPassword(this.keyManagerPassword);
            }
        }
        if (this.trustStorePath != null) {
            Optional<KeyStore> pemTrustStore = ReloadableSslContextFactoryProvider.tryLoadPemTrustStore(this.trustStorePath);
            if (pemTrustStore.isPresent()) {
                sslContextFactory.setTrustStore(pemTrustStore.get());
                sslContextFactory.setTrustStorePassword("");
            } else {
                sslContextFactory.setTrustStorePath(this.trustStorePath);
                sslContextFactory.setTrustStorePassword(this.trustStorePassword);
            }
        }
        sslContextFactory.setIncludeCipherSuites(this.includedCipherSuites.toArray(new String[0]));
        sslContextFactory.setExcludeCipherSuites(this.excludedCipherSuites.toArray(new String[0]));
        sslContextFactory.setSecureRandomAlgorithm(this.secureRandomAlgorithm);
        switch (this.clientCertificate) {
            case NONE: {
                break;
            }
            case REQUESTED: {
                sslContextFactory.setWantClientAuth(true);
                break;
            }
            case REQUIRED: {
                sslContextFactory.setNeedClientAuth(true);
                break;
            }
            default: {
                throw new IllegalArgumentException("Unsupported client certificate value: " + (Object)((Object)this.clientCertificate));
            }
        }
        sslContextFactory.setSslSessionTimeout(this.sslSessionTimeoutSeconds);
        sslContextFactory.setSslSessionCacheSize(this.sslSessionCacheSize);
        return sslContextFactory;
    }

    private static Optional<KeyStore> tryLoadPemKeyStore(String path, String password) {
        File keyStoreFile = new File(path);
        try {
            if (!PemReader.isPem((File)keyStoreFile)) {
                return Optional.empty();
            }
        }
        catch (IOException e) {
            throw new IllegalArgumentException("Error reading key store file: " + keyStoreFile, e);
        }
        try {
            return Optional.of(PemReader.loadKeyStore((File)keyStoreFile, (File)keyStoreFile, Optional.ofNullable(password)));
        }
        catch (IOException | GeneralSecurityException e) {
            throw new IllegalArgumentException("Error loading PEM key store: " + keyStoreFile, e);
        }
    }

    private static Optional<KeyStore> tryLoadPemTrustStore(String path) {
        File trustStoreFile = new File(path);
        try {
            if (!PemReader.isPem((File)trustStoreFile)) {
                return Optional.empty();
            }
        }
        catch (IOException e) {
            throw new IllegalArgumentException("Error reading trust store file: " + trustStoreFile, e);
        }
        try {
            if (PemReader.readCertificateChain((File)trustStoreFile).isEmpty()) {
                throw new IllegalArgumentException("PEM trust store file does not contain any certificates: " + trustStoreFile);
            }
            return Optional.of(PemReader.loadTrustStore((File)trustStoreFile));
        }
        catch (IOException | GeneralSecurityException e) {
            throw new IllegalArgumentException("Error loading PEM trust store: " + trustStoreFile, e);
        }
    }

    public SslContextFactory.Server getSslContextFactory() {
        return this.sslContextFactory;
    }

    private synchronized void reload() {
        try {
            SslContextFactory.Server updatedFactory = this.buildContextFactory();
            updatedFactory.start();
            this.sslContextFactory.reload(factory -> factory.setSslContext(updatedFactory.getSslContext()));
        }
        catch (Exception e) {
            log.warn((Throwable)e, "Unable to reload SslContext.");
        }
    }
}

