;; Copyright (c) Cognitect, Inc.
;; All rights reserved.

(ns ^{:doc "Tools for parsing and producing s3 access-key-ids
used by Datomic."}
  datomic.s3-access-keys
  (:require [clojure.set :as set]))

(set! *warn-on-reflection* true)

(def ^:private key-id-parsers
  {#"(.*)/([^/]+)/datomic/access/admin/([^/]+)"
   (fn [[_ prefix system key-name]]
     {:type :admin :prefix prefix :system system :key-name key-name})

   #"(.+)/([^/]+)/datomic/access/dbs/catalog/read/([^/]+)"
   (fn [[_ prefix system key-name]]
     {:type :catalog-read :prefix prefix :system system :key-name key-name})
   
   #"(.+)/([^/]+)/datomic/access/dbs/db/([^/]+)/([^/]+)"
   (fn [[_ prefix system db-name key-name]]
     {:type :db-write :prefix prefix :system system :db-name db-name :key-name key-name})
   
   #"(.+)/([^/]+)/datomic/access/dbs/db/([^/]+)/read/([^/]+)"
   (fn [[_ prefix system db-name key-name]]
     {:type :db-read :prefix prefix :system system :db-name db-name :key-name key-name})})

(defn parse-access-key-id
  "Convert a valid access-key-id into an access-key. Opposite of create-access-key-id."
  [s]
  (reduce-kv
   (fn [nomatch re f]
     (when-let [match (re-matches re s)]
       (reduced (f match))))
   nil
   key-id-parsers))

(defn- known-keys
  [m]
  (set/intersection #{:prefix :system :key-name :db-name}
                    (into #{} (keys m))))

(defmulti create-access-key-id
  "Convert a valid access-key map into an access-key-id, opposite of parse-access-key-id."
  (fn [arg] [(:type arg) (known-keys arg)]))
(defmethod create-access-key-id :default [_] nil)

(defmethod create-access-key-id [:admin #{:prefix :system :key-name}]
  [{:keys [prefix system key-name]}]
  (format "%s/%s/datomic/access/admin/%s" prefix system key-name))
(defmethod create-access-key-id [:catalog-read #{:prefix :system :key-name}]
  [{:keys [prefix system key-name]}]
  (format "%s/%s/datomic/access/dbs/catalog/read/%s" prefix system key-name))
(defmethod create-access-key-id [:db-read #{:prefix :system :key-name :db-name}]
  [{:keys [prefix system key-name db-name]}]
  (format "%s/%s/datomic/access/dbs/db/%s/read/%s" prefix system db-name key-name))
(defmethod create-access-key-id [:db-write #{:prefix :system :key-name :db-name}]
  [{:keys [prefix system key-name db-name]}]
  (format "%s/%s/datomic/access/dbs/db/%s/%s" prefix system db-name key-name))


