;; Copyright (c) Cognitect, Inc.
;; All rights reserved.

(ns cognitect.aws.dynamodb.specs
  (:require [clojure.spec.alpha :as s] [clojure.spec.gen.alpha :as gen]))

(s/def :cognitect.aws/client map?)

(s/def :core.async/channel any?)

(s/def :cognitect.aws.dynamodb/BooleanObject boolean?)

(s/def
  :cognitect.aws.dynamodb/ReturnItemCollectionMetrics
  (s/spec string? :gen #(s/gen #{"SIZE" "NONE"})))

(s/def
  :cognitect.aws.dynamodb/KinesisDataStreamDestinations
  (s/coll-of :cognitect.aws.dynamodb/KinesisDataStreamDestination))

(s/def
  :cognitect.aws.dynamodb/TransactGetItem
  (s/keys :req-un [:cognitect.aws.dynamodb.TransactGetItem/Get]))

(s/def
  :cognitect.aws.dynamodb/CloudWatchLogGroupArn
  (s/spec
    (s/and string? #(<= 1 (count %) 1024))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 1 1024) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.dynamodb/StreamViewType
  (s/spec string? :gen #(s/gen #{"NEW_IMAGE" "NEW_AND_OLD_IMAGES" "OLD_IMAGE" "KEYS_ONLY"})))

(s/def
  :cognitect.aws.dynamodb/BatchStatementResponse
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.BatchStatementResponse/TableName
     :cognitect.aws.dynamodb.BatchStatementResponse/Error
     :cognitect.aws.dynamodb.BatchStatementResponse/Item]))

(s/def
  :cognitect.aws.dynamodb/ReturnConsumedCapacity
  (s/spec string? :gen #(s/gen #{"NONE" "TOTAL" "INDEXES"})))

(s/def
  :cognitect.aws.dynamodb/AutoScalingSettingsDescription
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.AutoScalingSettingsDescription/MinimumUnits
     :cognitect.aws.dynamodb.AutoScalingSettingsDescription/AutoScalingDisabled
     :cognitect.aws.dynamodb.AutoScalingSettingsDescription/MaximumUnits
     :cognitect.aws.dynamodb.AutoScalingSettingsDescription/AutoScalingRoleArn
     :cognitect.aws.dynamodb.AutoScalingSettingsDescription/ScalingPolicies]))

(s/def
  :cognitect.aws.dynamodb/AttributeValueList
  (s/coll-of :cognitect.aws.dynamodb/AttributeValue))

(s/def
  :cognitect.aws.dynamodb/ListExportsMaxLimit
  (s/spec (s/and int? #(<= 1 % 25)) :gen #(gen/choose 1 25)))

(s/def
  :cognitect.aws.dynamodb/Update
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.Update/Key
     :cognitect.aws.dynamodb.Update/UpdateExpression
     :cognitect.aws.dynamodb.Update/TableName]
    :opt-un
    [:cognitect.aws.dynamodb.Update/ReturnValuesOnConditionCheckFailure
     :cognitect.aws.dynamodb.Update/ConditionExpression
     :cognitect.aws.dynamodb.Update/ExpressionAttributeNames
     :cognitect.aws.dynamodb.Update/ExpressionAttributeValues]))

(s/def
  :cognitect.aws.dynamodb/DeleteBackupInput
  (s/keys :req-un [:cognitect.aws.dynamodb.DeleteBackupInput/BackupArn]))

(s/def
  :cognitect.aws.dynamodb/ContributorInsightsRule
  (s/spec
    #(re-matches (re-pattern "[A-Za-z0-9][A-Za-z0-9\\-\\_\\.]{0,126}[A-Za-z0-9]") %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.dynamodb/ImportSummary
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.ImportSummary/EndTime
     :cognitect.aws.dynamodb.ImportSummary/ImportStatus
     :cognitect.aws.dynamodb.ImportSummary/StartTime
     :cognitect.aws.dynamodb.ImportSummary/ImportArn
     :cognitect.aws.dynamodb.ImportSummary/CloudWatchLogGroupArn
     :cognitect.aws.dynamodb.ImportSummary/TableArn
     :cognitect.aws.dynamodb.ImportSummary/S3BucketSource
     :cognitect.aws.dynamodb.ImportSummary/InputFormat]))

(s/def
  :cognitect.aws.dynamodb/ListImportsOutput
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.ListImportsOutput/ImportSummaryList
     :cognitect.aws.dynamodb.ListImportsOutput/NextToken]))

(s/def
  :cognitect.aws.dynamodb/DescribeKinesisStreamingDestinationOutput
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.DescribeKinesisStreamingDestinationOutput/KinesisDataStreamDestinations
     :cognitect.aws.dynamodb.DescribeKinesisStreamingDestinationOutput/TableName]))

(s/def
  :cognitect.aws.dynamodb/BackupSizeBytes
  (s/spec (s/and int? #(<= 0 %)) :gen #(gen/choose 0 Long/MAX_VALUE)))

(s/def
  :cognitect.aws.dynamodb/ExpressionAttributeNameMap
  (s/map-of
    :cognitect.aws.dynamodb/ExpressionAttributeNameVariable
    :cognitect.aws.dynamodb/AttributeName))

(s/def
  :cognitect.aws.dynamodb/ListContributorInsightsInput
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.ListContributorInsightsInput/NextToken
     :cognitect.aws.dynamodb.ListContributorInsightsInput/TableName
     :cognitect.aws.dynamodb.ListContributorInsightsInput/MaxResults]))

(s/def
  :cognitect.aws.dynamodb/DescribeContinuousBackupsInput
  (s/keys :req-un [:cognitect.aws.dynamodb.DescribeContinuousBackupsInput/TableName]))

(s/def
  :cognitect.aws.dynamodb/ListBackupsOutput
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.ListBackupsOutput/BackupSummaries
     :cognitect.aws.dynamodb.ListBackupsOutput/LastEvaluatedBackupArn]))

(s/def
  :cognitect.aws.dynamodb/DescribeTableReplicaAutoScalingInput
  (s/keys :req-un [:cognitect.aws.dynamodb.DescribeTableReplicaAutoScalingInput/TableName]))

(s/def
  :cognitect.aws.dynamodb/DeleteReplicationGroupMemberAction
  (s/keys :req-un [:cognitect.aws.dynamodb.DeleteReplicationGroupMemberAction/RegionName]))

(s/def
  :cognitect.aws.dynamodb/S3BucketSource
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.S3BucketSource/S3Bucket]
    :opt-un
    [:cognitect.aws.dynamodb.S3BucketSource/S3BucketOwner
     :cognitect.aws.dynamodb.S3BucketSource/S3KeyPrefix]))

(s/def
  :cognitect.aws.dynamodb/DescribeGlobalTableOutput
  (s/keys :opt-un [:cognitect.aws.dynamodb.DescribeGlobalTableOutput/GlobalTableDescription]))

(s/def
  :cognitect.aws.dynamodb/ImportedItemCount
  (s/spec (s/and int? #(<= 0 %)) :gen #(gen/choose 0 Long/MAX_VALUE)))

(s/def
  :cognitect.aws.dynamodb/UpdateItemOutput
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.UpdateItemOutput/ItemCollectionMetrics
     :cognitect.aws.dynamodb.UpdateItemOutput/Attributes
     :cognitect.aws.dynamodb.UpdateItemOutput/ConsumedCapacity]))

(s/def :cognitect.aws.dynamodb/BooleanAttributeValue boolean?)

(s/def
  :cognitect.aws.dynamodb/ListExportsInput
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.ListExportsInput/NextToken
     :cognitect.aws.dynamodb.ListExportsInput/TableArn
     :cognitect.aws.dynamodb.ListExportsInput/MaxResults]))

(s/def
  :cognitect.aws.dynamodb/LocalSecondaryIndex
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.LocalSecondaryIndex/IndexName
     :cognitect.aws.dynamodb.LocalSecondaryIndex/KeySchema
     :cognitect.aws.dynamodb.LocalSecondaryIndex/Projection]))

(s/def
  :cognitect.aws.dynamodb/ListGlobalTablesOutput
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.ListGlobalTablesOutput/GlobalTables
     :cognitect.aws.dynamodb.ListGlobalTablesOutput/LastEvaluatedGlobalTableName]))

(s/def
  :cognitect.aws.dynamodb/ReplicaDescription
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.ReplicaDescription/ReplicaStatus
     :cognitect.aws.dynamodb.ReplicaDescription/ProvisionedThroughputOverride
     :cognitect.aws.dynamodb.ReplicaDescription/ReplicaStatusDescription
     :cognitect.aws.dynamodb.ReplicaDescription/ReplicaTableClassSummary
     :cognitect.aws.dynamodb.ReplicaDescription/KMSMasterKeyId
     :cognitect.aws.dynamodb.ReplicaDescription/RegionName
     :cognitect.aws.dynamodb.ReplicaDescription/ReplicaStatusPercentProgress
     :cognitect.aws.dynamodb.ReplicaDescription/GlobalSecondaryIndexes
     :cognitect.aws.dynamodb.ReplicaDescription/ReplicaInaccessibleDateTime]))

(s/def
  :cognitect.aws.dynamodb/CreateBackupOutput
  (s/keys :opt-un [:cognitect.aws.dynamodb.CreateBackupOutput/BackupDetails]))

(s/def
  :cognitect.aws.dynamodb/Tag
  (s/keys :req-un [:cognitect.aws.dynamodb.Tag/Key :cognitect.aws.dynamodb.Tag/Value]))

(s/def
  :cognitect.aws.dynamodb/ScanTotalSegments
  (s/spec (s/and int? #(<= 1 % 1000000)) :gen #(gen/choose 1 1000000)))

(s/def
  :cognitect.aws.dynamodb/ListTagsOfResourceOutput
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.ListTagsOfResourceOutput/Tags
     :cognitect.aws.dynamodb.ListTagsOfResourceOutput/NextToken]))

(s/def :cognitect.aws.dynamodb/NumberAttributeValue string?)

(s/def
  :cognitect.aws.dynamodb/BatchExecuteStatementOutput
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.BatchExecuteStatementOutput/Responses
     :cognitect.aws.dynamodb.BatchExecuteStatementOutput/ConsumedCapacity]))

(s/def
  :cognitect.aws.dynamodb/ReplicaSettingsUpdate
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.ReplicaSettingsUpdate/RegionName]
    :opt-un
    [:cognitect.aws.dynamodb.ReplicaSettingsUpdate/ReplicaGlobalSecondaryIndexSettingsUpdate
     :cognitect.aws.dynamodb.ReplicaSettingsUpdate/ReplicaProvisionedReadCapacityAutoScalingSettingsUpdate
     :cognitect.aws.dynamodb.ReplicaSettingsUpdate/ReplicaTableClass
     :cognitect.aws.dynamodb.ReplicaSettingsUpdate/ReplicaProvisionedReadCapacityUnits]))

(s/def
  :cognitect.aws.dynamodb/SSESpecification
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.SSESpecification/Enabled
     :cognitect.aws.dynamodb.SSESpecification/KMSMasterKeyId
     :cognitect.aws.dynamodb.SSESpecification/SSEType]))

(s/def :cognitect.aws.dynamodb/GlobalTableList (s/coll-of :cognitect.aws.dynamodb/GlobalTable))

(s/def
  :cognitect.aws.dynamodb/TableAutoScalingDescription
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.TableAutoScalingDescription/Replicas
     :cognitect.aws.dynamodb.TableAutoScalingDescription/TableStatus
     :cognitect.aws.dynamodb.TableAutoScalingDescription/TableName]))

(s/def
  :cognitect.aws.dynamodb/ReplicaGlobalSecondaryIndexSettingsUpdate
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndexSettingsUpdate/IndexName]
    :opt-un
    [:cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndexSettingsUpdate/ProvisionedReadCapacityUnits
     :cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndexSettingsUpdate/ProvisionedReadCapacityAutoScalingSettingsUpdate]))

(s/def
  :cognitect.aws.dynamodb/UpdateTableReplicaAutoScalingInput
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.UpdateTableReplicaAutoScalingInput/TableName]
    :opt-un
    [:cognitect.aws.dynamodb.UpdateTableReplicaAutoScalingInput/GlobalSecondaryIndexUpdates
     :cognitect.aws.dynamodb.UpdateTableReplicaAutoScalingInput/ProvisionedWriteCapacityAutoScalingUpdate
     :cognitect.aws.dynamodb.UpdateTableReplicaAutoScalingInput/ReplicaUpdates]))

(s/def :cognitect.aws.dynamodb/RegionName string?)

(s/def
  :cognitect.aws.dynamodb/ReplicaAutoScalingUpdate
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.ReplicaAutoScalingUpdate/RegionName]
    :opt-un
    [:cognitect.aws.dynamodb.ReplicaAutoScalingUpdate/ReplicaGlobalSecondaryIndexUpdates
     :cognitect.aws.dynamodb.ReplicaAutoScalingUpdate/ReplicaProvisionedReadCapacityAutoScalingUpdate]))

(s/def
  :cognitect.aws.dynamodb/ReplicaUpdate
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.ReplicaUpdate/Delete :cognitect.aws.dynamodb.ReplicaUpdate/Create]))

(s/def :cognitect.aws.dynamodb/ReplicaStatusDescription string?)

(s/def
  :cognitect.aws.dynamodb/DescribeImportOutput
  (s/keys :req-un [:cognitect.aws.dynamodb.DescribeImportOutput/ImportTableDescription]))

(s/def
  :cognitect.aws.dynamodb/UpdateContributorInsightsOutput
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.UpdateContributorInsightsOutput/IndexName
     :cognitect.aws.dynamodb.UpdateContributorInsightsOutput/ContributorInsightsStatus
     :cognitect.aws.dynamodb.UpdateContributorInsightsOutput/TableName]))

(s/def
  :cognitect.aws.dynamodb/KeySchemaAttributeName
  (s/spec
    (s/and string? #(<= 1 (count %) 255))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 1 255) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.dynamodb/DescribeContinuousBackupsOutput
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.DescribeContinuousBackupsOutput/ContinuousBackupsDescription]))

(s/def
  :cognitect.aws.dynamodb/DescribeBackupOutput
  (s/keys :opt-un [:cognitect.aws.dynamodb.DescribeBackupOutput/BackupDescription]))

(s/def
  :cognitect.aws.dynamodb/GlobalSecondaryIndexUpdateList
  (s/coll-of :cognitect.aws.dynamodb/GlobalSecondaryIndexUpdate))

(s/def :cognitect.aws.dynamodb/ReplicaStatusPercentProgress string?)

(s/def
  :cognitect.aws.dynamodb/CsvOptions
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.CsvOptions/Delimiter :cognitect.aws.dynamodb.CsvOptions/HeaderList]))

(s/def
  :cognitect.aws.dynamodb/AutoScalingSettingsUpdate
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.AutoScalingSettingsUpdate/MinimumUnits
     :cognitect.aws.dynamodb.AutoScalingSettingsUpdate/AutoScalingDisabled
     :cognitect.aws.dynamodb.AutoScalingSettingsUpdate/MaximumUnits
     :cognitect.aws.dynamodb.AutoScalingSettingsUpdate/ScalingPolicyUpdate
     :cognitect.aws.dynamodb.AutoScalingSettingsUpdate/AutoScalingRoleArn]))

(s/def
  :cognitect.aws.dynamodb/AttributeNameList
  (s/coll-of :cognitect.aws.dynamodb/AttributeName :min-count 1))

(s/def
  :cognitect.aws.dynamodb/ImportTableDescription
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.ImportTableDescription/EndTime
     :cognitect.aws.dynamodb.ImportTableDescription/ImportedItemCount
     :cognitect.aws.dynamodb.ImportTableDescription/ImportStatus
     :cognitect.aws.dynamodb.ImportTableDescription/InputFormatOptions
     :cognitect.aws.dynamodb.ImportTableDescription/ErrorCount
     :cognitect.aws.dynamodb.ImportTableDescription/StartTime
     :cognitect.aws.dynamodb.ImportTableDescription/ImportArn
     :cognitect.aws.dynamodb.ImportTableDescription/TableId
     :cognitect.aws.dynamodb.ImportTableDescription/ProcessedSizeBytes
     :cognitect.aws.dynamodb.ImportTableDescription/CloudWatchLogGroupArn
     :cognitect.aws.dynamodb.ImportTableDescription/ProcessedItemCount
     :cognitect.aws.dynamodb.ImportTableDescription/TableCreationParameters
     :cognitect.aws.dynamodb.ImportTableDescription/FailureMessage
     :cognitect.aws.dynamodb.ImportTableDescription/InputCompressionType
     :cognitect.aws.dynamodb.ImportTableDescription/TableArn
     :cognitect.aws.dynamodb.ImportTableDescription/S3BucketSource
     :cognitect.aws.dynamodb.ImportTableDescription/InputFormat
     :cognitect.aws.dynamodb.ImportTableDescription/FailureCode
     :cognitect.aws.dynamodb.ImportTableDescription/ClientToken]))

(s/def
  :cognitect.aws.dynamodb/ParameterizedStatement
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.ParameterizedStatement/Statement]
    :opt-un
    [:cognitect.aws.dynamodb.ParameterizedStatement/Parameters
     :cognitect.aws.dynamodb.ParameterizedStatement/ReturnValuesOnConditionCheckFailure]))

(s/def
  :cognitect.aws.dynamodb/ReplicaDescriptionList
  (s/coll-of :cognitect.aws.dynamodb/ReplicaDescription))

(s/def
  :cognitect.aws.dynamodb/TableStatus
  (s/spec
    string?
    :gen
    #(s/gen
      #{"ARCHIVED"
        "UPDATING"
        "DELETING"
        "CREATING"
        "ACTIVE"
        "ARCHIVING"
        "INACCESSIBLE_ENCRYPTION_CREDENTIALS"})))

(s/def
  :cognitect.aws.dynamodb/ReplicaGlobalSecondaryIndexSettingsDescription
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndexSettingsDescription/IndexName]
    :opt-un
    [:cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndexSettingsDescription/IndexStatus
     :cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndexSettingsDescription/ProvisionedReadCapacityUnits
     :cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndexSettingsDescription/ProvisionedWriteCapacityUnits
     :cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndexSettingsDescription/ProvisionedWriteCapacityAutoScalingSettings
     :cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndexSettingsDescription/ProvisionedReadCapacityAutoScalingSettings]))

(s/def
  :cognitect.aws.dynamodb/GlobalSecondaryIndexDescriptionList
  (s/coll-of :cognitect.aws.dynamodb/GlobalSecondaryIndexDescription))

(s/def
  :cognitect.aws.dynamodb/CreateTableOutput
  (s/keys :opt-un [:cognitect.aws.dynamodb.CreateTableOutput/TableDescription]))

(s/def
  :cognitect.aws.dynamodb/ListTablesOutput
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.ListTablesOutput/TableNames
     :cognitect.aws.dynamodb.ListTablesOutput/LastEvaluatedTableName]))

(s/def
  :cognitect.aws.dynamodb/AutoScalingRoleArn
  (s/spec
    #(re-matches
      (re-pattern "[\\u0020-\\uD7FF\\uE000-\\uFFFD\\uD800\\uDC00-\\uDBFF\\uDFFF\\r\\n\\t]*")
      %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.dynamodb/BatchWriteItemOutput
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.BatchWriteItemOutput/ItemCollectionMetrics
     :cognitect.aws.dynamodb.BatchWriteItemOutput/UnprocessedItems
     :cognitect.aws.dynamodb.BatchWriteItemOutput/ConsumedCapacity]))

(s/def
  :cognitect.aws.dynamodb/PutRequest
  (s/keys :req-un [:cognitect.aws.dynamodb.PutRequest/Item]))

(s/def :cognitect.aws.dynamodb/NextTokenString string?)

(s/def
  :cognitect.aws.dynamodb/ReplicaGlobalSecondaryIndex
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndex/IndexName]
    :opt-un
    [:cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndex/ProvisionedThroughputOverride]))

(s/def
  :cognitect.aws.dynamodb/ContributorInsightsRuleList
  (s/coll-of :cognitect.aws.dynamodb/ContributorInsightsRule))

(s/def
  :cognitect.aws.dynamodb/ParameterizedStatements
  (s/coll-of :cognitect.aws.dynamodb/ParameterizedStatement :min-count 1 :max-count 100))

(s/def
  :cognitect.aws.dynamodb/QueryOutput
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.QueryOutput/Count
     :cognitect.aws.dynamodb.QueryOutput/LastEvaluatedKey
     :cognitect.aws.dynamodb.QueryOutput/ScannedCount
     :cognitect.aws.dynamodb.QueryOutput/ConsumedCapacity
     :cognitect.aws.dynamodb.QueryOutput/Items]))

(s/def :cognitect.aws.dynamodb/ExportSummaries (s/coll-of :cognitect.aws.dynamodb/ExportSummary))

(s/def
  :cognitect.aws.dynamodb/Endpoint
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.Endpoint/Address
     :cognitect.aws.dynamodb.Endpoint/CachePeriodInMinutes]))

(s/def :cognitect.aws.dynamodb/ExpressionAttributeNameVariable string?)

(s/def
  :cognitect.aws.dynamodb/ListContributorInsightsOutput
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.ListContributorInsightsOutput/NextToken
     :cognitect.aws.dynamodb.ListContributorInsightsOutput/ContributorInsightsSummaries]))

(s/def
  :cognitect.aws.dynamodb/SSEStatus
  (s/spec string? :gen #(s/gen #{"DISABLED" "DISABLING" "ENABLING" "UPDATING" "ENABLED"})))

(s/def
  :cognitect.aws.dynamodb/PositiveIntegerObject
  (s/spec (s/and int? #(<= 1 %)) :gen #(gen/choose 1 Long/MAX_VALUE)))

(s/def :cognitect.aws.dynamodb/StreamEnabled boolean?)

(s/def
  :cognitect.aws.dynamodb/BilledSizeBytes
  (s/spec (s/and int? #(<= 0 %)) :gen #(gen/choose 0 Long/MAX_VALUE)))

(s/def
  :cognitect.aws.dynamodb/ReplicaGlobalSecondaryIndexAutoScalingUpdateList
  (s/coll-of :cognitect.aws.dynamodb/ReplicaGlobalSecondaryIndexAutoScalingUpdate))

(s/def
  :cognitect.aws.dynamodb/ReplicaSettingsDescription
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.ReplicaSettingsDescription/RegionName]
    :opt-un
    [:cognitect.aws.dynamodb.ReplicaSettingsDescription/ReplicaStatus
     :cognitect.aws.dynamodb.ReplicaSettingsDescription/ReplicaProvisionedReadCapacityAutoScalingSettings
     :cognitect.aws.dynamodb.ReplicaSettingsDescription/ReplicaGlobalSecondaryIndexSettings
     :cognitect.aws.dynamodb.ReplicaSettingsDescription/ReplicaProvisionedWriteCapacityUnits
     :cognitect.aws.dynamodb.ReplicaSettingsDescription/ReplicaTableClassSummary
     :cognitect.aws.dynamodb.ReplicaSettingsDescription/ReplicaProvisionedReadCapacityUnits
     :cognitect.aws.dynamodb.ReplicaSettingsDescription/ReplicaProvisionedWriteCapacityAutoScalingSettings
     :cognitect.aws.dynamodb.ReplicaSettingsDescription/ReplicaBillingModeSummary]))

(s/def
  :cognitect.aws.dynamodb/GlobalTableDescription
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.GlobalTableDescription/GlobalTableName
     :cognitect.aws.dynamodb.GlobalTableDescription/ReplicationGroup
     :cognitect.aws.dynamodb.GlobalTableDescription/GlobalTableArn
     :cognitect.aws.dynamodb.GlobalTableDescription/CreationDateTime
     :cognitect.aws.dynamodb.GlobalTableDescription/GlobalTableStatus]))

(s/def
  :cognitect.aws.dynamodb/ExpectedAttributeValue
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.ExpectedAttributeValue/Exists
     :cognitect.aws.dynamodb.ExpectedAttributeValue/ComparisonOperator
     :cognitect.aws.dynamodb.ExpectedAttributeValue/Value
     :cognitect.aws.dynamodb.ExpectedAttributeValue/AttributeValueList]))

(s/def
  :cognitect.aws.dynamodb/AutoScalingPolicyDescriptionList
  (s/coll-of :cognitect.aws.dynamodb/AutoScalingPolicyDescription))

(s/def
  :cognitect.aws.dynamodb/CsvDelimiter
  (s/spec #(re-matches (re-pattern "[,;:|\\t ]") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.dynamodb/KinesisDataStreamDestination
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.KinesisDataStreamDestination/DestinationStatusDescription
     :cognitect.aws.dynamodb.KinesisDataStreamDestination/StreamArn
     :cognitect.aws.dynamodb.KinesisDataStreamDestination/DestinationStatus]))

(s/def
  :cognitect.aws.dynamodb/GlobalTableGlobalSecondaryIndexSettingsUpdateList
  (s/coll-of
    :cognitect.aws.dynamodb/GlobalTableGlobalSecondaryIndexSettingsUpdate
    :min-count
    1
    :max-count
    20))

(s/def
  :cognitect.aws.dynamodb/ProvisionedThroughputDescription
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.ProvisionedThroughputDescription/WriteCapacityUnits
     :cognitect.aws.dynamodb.ProvisionedThroughputDescription/NumberOfDecreasesToday
     :cognitect.aws.dynamodb.ProvisionedThroughputDescription/LastDecreaseDateTime
     :cognitect.aws.dynamodb.ProvisionedThroughputDescription/ReadCapacityUnits
     :cognitect.aws.dynamodb.ProvisionedThroughputDescription/LastIncreaseDateTime]))

(s/def
  :cognitect.aws.dynamodb/ExecuteTransactionInput
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.ExecuteTransactionInput/TransactStatements]
    :opt-un
    [:cognitect.aws.dynamodb.ExecuteTransactionInput/ClientRequestToken
     :cognitect.aws.dynamodb.ExecuteTransactionInput/ReturnConsumedCapacity]))

(s/def :cognitect.aws.dynamodb/SSEEnabled boolean?)

(s/def
  :cognitect.aws.dynamodb/PartiQLStatement
  (s/spec
    (s/and string? #(<= 1 (count %) 8192))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 1 8192) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.dynamodb/TagValueString
  (s/spec
    (s/and string? #(<= 0 (count %) 256))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 0 256) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.dynamodb/BinarySetAttributeValue
  (s/coll-of :cognitect.aws.dynamodb/BinaryAttributeValue))

(s/def
  :cognitect.aws.dynamodb/TableClassSummary
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.TableClassSummary/TableClass
     :cognitect.aws.dynamodb.TableClassSummary/LastUpdateDateTime]))

(s/def
  :cognitect.aws.dynamodb/FailureException
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.FailureException/ExceptionDescription
     :cognitect.aws.dynamodb.FailureException/ExceptionName]))

(s/def
  :cognitect.aws.dynamodb/TagKeyString
  (s/spec
    (s/and string? #(<= 1 (count %) 128))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 1 128) #(gen/vector (gen/char-alpha) %))))))

(s/def :cognitect.aws.dynamodb/ExceptionName string?)

(s/def
  :cognitect.aws.dynamodb/DescribeTableInput
  (s/keys :req-un [:cognitect.aws.dynamodb.DescribeTableInput/TableName]))

(s/def :cognitect.aws.dynamodb/TagList (s/coll-of :cognitect.aws.dynamodb/Tag))

(s/def
  :cognitect.aws.dynamodb/DescribeTableOutput
  (s/keys :opt-un [:cognitect.aws.dynamodb.DescribeTableOutput/Table]))

(s/def
  :cognitect.aws.dynamodb/BillingMode
  (s/spec string? :gen #(s/gen #{"PROVISIONED" "PAY_PER_REQUEST"})))

(s/def
  :cognitect.aws.dynamodb/ExpressionAttributeValueMap
  (s/map-of
    :cognitect.aws.dynamodb/ExpressionAttributeValueVariable
    :cognitect.aws.dynamodb/AttributeValue))

(s/def
  :cognitect.aws.dynamodb/TableClass
  (s/spec string? :gen #(s/gen #{"STANDARD_INFREQUENT_ACCESS" "STANDARD"})))

(s/def
  :cognitect.aws.dynamodb/ListContributorInsightsLimit
  (s/spec (s/and int? #(>= 100 %)) :gen #(gen/choose Long/MIN_VALUE 100)))

(s/def
  :cognitect.aws.dynamodb/RestoreSummary
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.RestoreSummary/RestoreDateTime
     :cognitect.aws.dynamodb.RestoreSummary/RestoreInProgress]
    :opt-un
    [:cognitect.aws.dynamodb.RestoreSummary/SourceBackupArn
     :cognitect.aws.dynamodb.RestoreSummary/SourceTableArn]))

(s/def
  :cognitect.aws.dynamodb/BackupSummary
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.BackupSummary/BackupStatus
     :cognitect.aws.dynamodb.BackupSummary/TableId
     :cognitect.aws.dynamodb.BackupSummary/BackupCreationDateTime
     :cognitect.aws.dynamodb.BackupSummary/BackupName
     :cognitect.aws.dynamodb.BackupSummary/BackupExpiryDateTime
     :cognitect.aws.dynamodb.BackupSummary/BackupArn
     :cognitect.aws.dynamodb.BackupSummary/TableName
     :cognitect.aws.dynamodb.BackupSummary/TableArn
     :cognitect.aws.dynamodb.BackupSummary/BackupType
     :cognitect.aws.dynamodb.BackupSummary/BackupSizeBytes]))

(s/def
  :cognitect.aws.dynamodb/S3BucketOwner
  (s/spec #(re-matches (re-pattern "[0-9]{12}") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.dynamodb/AttributeUpdates
  (s/map-of :cognitect.aws.dynamodb/AttributeName :cognitect.aws.dynamodb/AttributeValueUpdate))

(s/def
  :cognitect.aws.dynamodb/DoubleObject
  (s/spec double? :gen #(gen/double* {:infinite? false, :NaN? false})))

(s/def
  :cognitect.aws.dynamodb/DescribeGlobalTableInput
  (s/keys :req-un [:cognitect.aws.dynamodb.DescribeGlobalTableInput/GlobalTableName]))

(s/def
  :cognitect.aws.dynamodb/Key
  (s/map-of :cognitect.aws.dynamodb/AttributeName :cognitect.aws.dynamodb/AttributeValue))

(s/def
  :cognitect.aws.dynamodb/BatchExecuteStatementInput
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.BatchExecuteStatementInput/Statements]
    :opt-un
    [:cognitect.aws.dynamodb.BatchExecuteStatementInput/ReturnConsumedCapacity]))

(s/def
  :cognitect.aws.dynamodb/GlobalSecondaryIndexAutoScalingUpdate
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.GlobalSecondaryIndexAutoScalingUpdate/IndexName
     :cognitect.aws.dynamodb.GlobalSecondaryIndexAutoScalingUpdate/ProvisionedWriteCapacityAutoScalingUpdate]))

(s/def
  :cognitect.aws.dynamodb/DestinationStatus
  (s/spec string? :gen #(s/gen #{"ENABLE_FAILED" "DISABLED" "DISABLING" "ENABLING" "ACTIVE"})))

(s/def
  :cognitect.aws.dynamodb/BatchGetItemOutput
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.BatchGetItemOutput/Responses
     :cognitect.aws.dynamodb.BatchGetItemOutput/UnprocessedKeys
     :cognitect.aws.dynamodb.BatchGetItemOutput/ConsumedCapacity]))

(s/def
  :cognitect.aws.dynamodb/CsvHeader
  (s/spec #(re-matches (re-pattern "[\\x20-\\x21\\x23-\\x2B\\x2D-\\x7E]*") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.dynamodb/TransactGetItemList
  (s/coll-of :cognitect.aws.dynamodb/TransactGetItem :min-count 1 :max-count 100))

(s/def
  :cognitect.aws.dynamodb/ContributorInsightsAction
  (s/spec string? :gen #(s/gen #{"DISABLE" "ENABLE"})))

(s/def
  :cognitect.aws.dynamodb/DeleteTableInput
  (s/keys :req-un [:cognitect.aws.dynamodb.DeleteTableInput/TableName]))

(s/def
  :cognitect.aws.dynamodb/ReplicaGlobalSecondaryIndexDescription
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndexDescription/IndexName
     :cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndexDescription/ProvisionedThroughputOverride]))

(s/def
  :cognitect.aws.dynamodb/AttributeAction
  (s/spec string? :gen #(s/gen #{"DELETE" "ADD" "PUT"})))

(s/def
  :cognitect.aws.dynamodb/S3Prefix
  (s/spec
    (s/and string? #(>= 1024 (count %)))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 0 1024) #(gen/vector (gen/char-alpha) %))))))

(s/def :cognitect.aws.dynamodb/FailureMessage string?)

(s/def
  :cognitect.aws.dynamodb/ImportStatus
  (s/spec string? :gen #(s/gen #{"COMPLETED" "IN_PROGRESS" "CANCELLING" "FAILED" "CANCELLED"})))

(s/def
  :cognitect.aws.dynamodb/ReplicaSettingsUpdateList
  (s/coll-of :cognitect.aws.dynamodb/ReplicaSettingsUpdate :min-count 1 :max-count 50))

(s/def
  :cognitect.aws.dynamodb/DescribeBackupInput
  (s/keys :req-un [:cognitect.aws.dynamodb.DescribeBackupInput/BackupArn]))

(s/def
  :cognitect.aws.dynamodb/ContributorInsightsSummary
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.ContributorInsightsSummary/IndexName
     :cognitect.aws.dynamodb.ContributorInsightsSummary/ContributorInsightsStatus
     :cognitect.aws.dynamodb.ContributorInsightsSummary/TableName]))

(s/def
  :cognitect.aws.dynamodb/QueryInput
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.QueryInput/TableName]
    :opt-un
    [:cognitect.aws.dynamodb.QueryInput/ScanIndexForward
     :cognitect.aws.dynamodb.QueryInput/IndexName
     :cognitect.aws.dynamodb.QueryInput/AttributesToGet
     :cognitect.aws.dynamodb.QueryInput/Limit
     :cognitect.aws.dynamodb.QueryInput/ConsistentRead
     :cognitect.aws.dynamodb.QueryInput/KeyConditionExpression
     :cognitect.aws.dynamodb.QueryInput/ReturnConsumedCapacity
     :cognitect.aws.dynamodb.QueryInput/ConditionalOperator
     :cognitect.aws.dynamodb.QueryInput/FilterExpression
     :cognitect.aws.dynamodb.QueryInput/ExclusiveStartKey
     :cognitect.aws.dynamodb.QueryInput/ExpressionAttributeNames
     :cognitect.aws.dynamodb.QueryInput/KeyConditions
     :cognitect.aws.dynamodb.QueryInput/QueryFilter
     :cognitect.aws.dynamodb.QueryInput/ProjectionExpression
     :cognitect.aws.dynamodb.QueryInput/Select
     :cognitect.aws.dynamodb.QueryInput/ExpressionAttributeValues]))

(s/def
  :cognitect.aws.dynamodb/ProvisionedThroughputOverride
  (s/keys :opt-un [:cognitect.aws.dynamodb.ProvisionedThroughputOverride/ReadCapacityUnits]))

(s/def
  :cognitect.aws.dynamodb/AutoScalingTargetTrackingScalingPolicyConfigurationDescription
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.AutoScalingTargetTrackingScalingPolicyConfigurationDescription/TargetValue]
    :opt-un
    [:cognitect.aws.dynamodb.AutoScalingTargetTrackingScalingPolicyConfigurationDescription/ScaleInCooldown
     :cognitect.aws.dynamodb.AutoScalingTargetTrackingScalingPolicyConfigurationDescription/DisableScaleIn
     :cognitect.aws.dynamodb.AutoScalingTargetTrackingScalingPolicyConfigurationDescription/ScaleOutCooldown]))

(s/def
  :cognitect.aws.dynamodb/BatchWriteItemRequestMap
  (s/map-of
    :cognitect.aws.dynamodb/TableName
    :cognitect.aws.dynamodb/WriteRequests
    :min-count
    1
    :max-count
    25))

(s/def
  :cognitect.aws.dynamodb/NonKeyAttributeName
  (s/spec
    (s/and string? #(<= 1 (count %) 255))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 1 255) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.dynamodb/BatchGetItemInput
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.BatchGetItemInput/RequestItems]
    :opt-un
    [:cognitect.aws.dynamodb.BatchGetItemInput/ReturnConsumedCapacity]))

(s/def
  :cognitect.aws.dynamodb/ImportTableOutput
  (s/keys :req-un [:cognitect.aws.dynamodb.ImportTableOutput/ImportTableDescription]))

(s/def
  :cognitect.aws.dynamodb/ExportTableToPointInTimeInput
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.ExportTableToPointInTimeInput/TableArn
     :cognitect.aws.dynamodb.ExportTableToPointInTimeInput/S3Bucket]
    :opt-un
    [:cognitect.aws.dynamodb.ExportTableToPointInTimeInput/S3SseKmsKeyId
     :cognitect.aws.dynamodb.ExportTableToPointInTimeInput/S3BucketOwner
     :cognitect.aws.dynamodb.ExportTableToPointInTimeInput/ExportTime
     :cognitect.aws.dynamodb.ExportTableToPointInTimeInput/S3SseAlgorithm
     :cognitect.aws.dynamodb.ExportTableToPointInTimeInput/S3Prefix
     :cognitect.aws.dynamodb.ExportTableToPointInTimeInput/ExportFormat
     :cognitect.aws.dynamodb.ExportTableToPointInTimeInput/ClientToken]))

(s/def
  :cognitect.aws.dynamodb/RestoreTableToPointInTimeOutput
  (s/keys :opt-un [:cognitect.aws.dynamodb.RestoreTableToPointInTimeOutput/TableDescription]))

(s/def
  :cognitect.aws.dynamodb/ItemCollectionMetricsMultiple
  (s/coll-of :cognitect.aws.dynamodb/ItemCollectionMetrics))

(s/def
  :cognitect.aws.dynamodb/BackupStatus
  (s/spec string? :gen #(s/gen #{"AVAILABLE" "CREATING" "DELETED"})))

(s/def
  :cognitect.aws.dynamodb/ImportNextToken
  (s/spec #(re-matches (re-pattern "([0-9a-f]{16})+") %) :gen #(gen/string)))

(s/def :cognitect.aws.dynamodb/ExportEndTime inst?)

(s/def
  :cognitect.aws.dynamodb/UpdateTimeToLiveOutput
  (s/keys :opt-un [:cognitect.aws.dynamodb.UpdateTimeToLiveOutput/TimeToLiveSpecification]))

(s/def
  :cognitect.aws.dynamodb/ExportDescription
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.ExportDescription/ExportManifest
     :cognitect.aws.dynamodb.ExportDescription/EndTime
     :cognitect.aws.dynamodb.ExportDescription/ExportStatus
     :cognitect.aws.dynamodb.ExportDescription/StartTime
     :cognitect.aws.dynamodb.ExportDescription/S3SseKmsKeyId
     :cognitect.aws.dynamodb.ExportDescription/TableId
     :cognitect.aws.dynamodb.ExportDescription/S3BucketOwner
     :cognitect.aws.dynamodb.ExportDescription/ExportArn
     :cognitect.aws.dynamodb.ExportDescription/FailureMessage
     :cognitect.aws.dynamodb.ExportDescription/BilledSizeBytes
     :cognitect.aws.dynamodb.ExportDescription/ExportTime
     :cognitect.aws.dynamodb.ExportDescription/ItemCount
     :cognitect.aws.dynamodb.ExportDescription/S3Bucket
     :cognitect.aws.dynamodb.ExportDescription/S3SseAlgorithm
     :cognitect.aws.dynamodb.ExportDescription/TableArn
     :cognitect.aws.dynamodb.ExportDescription/S3Prefix
     :cognitect.aws.dynamodb.ExportDescription/ExportFormat
     :cognitect.aws.dynamodb.ExportDescription/FailureCode
     :cognitect.aws.dynamodb.ExportDescription/ClientToken]))

(s/def
  :cognitect.aws.dynamodb/CsvHeaderList
  (s/coll-of :cognitect.aws.dynamodb/CsvHeader :min-count 1 :max-count 255))

(s/def
  :cognitect.aws.dynamodb/ComparisonOperator
  (s/spec
    string?
    :gen
    #(s/gen
      #{"NULL"
        "NE"
        "CONTAINS"
        "LT"
        "GE"
        "LE"
        "NOT_CONTAINS"
        "BEGINS_WITH"
        "NOT_NULL"
        "EQ"
        "GT"
        "IN"
        "BETWEEN"})))

(s/def
  :cognitect.aws.dynamodb/DescribeGlobalTableSettingsOutput
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.DescribeGlobalTableSettingsOutput/GlobalTableName
     :cognitect.aws.dynamodb.DescribeGlobalTableSettingsOutput/ReplicaSettings]))

(s/def :cognitect.aws.dynamodb/ExpressionAttributeValueVariable string?)

(s/def
  :cognitect.aws.dynamodb/TagResourceInput
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.TagResourceInput/ResourceArn
     :cognitect.aws.dynamodb.TagResourceInput/Tags]))

(s/def
  :cognitect.aws.dynamodb/BatchWriteItemInput
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.BatchWriteItemInput/RequestItems]
    :opt-un
    [:cognitect.aws.dynamodb.BatchWriteItemInput/ReturnConsumedCapacity
     :cognitect.aws.dynamodb.BatchWriteItemInput/ReturnItemCollectionMetrics]))

(s/def
  :cognitect.aws.dynamodb/BatchGetResponseMap
  (s/map-of :cognitect.aws.dynamodb/TableName :cognitect.aws.dynamodb/ItemList))

(s/def
  :cognitect.aws.dynamodb/ExportTableToPointInTimeOutput
  (s/keys :opt-un [:cognitect.aws.dynamodb.ExportTableToPointInTimeOutput/ExportDescription]))

(s/def
  :cognitect.aws.dynamodb/UpdateTimeToLiveInput
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.UpdateTimeToLiveInput/TableName
     :cognitect.aws.dynamodb.UpdateTimeToLiveInput/TimeToLiveSpecification]))

(s/def
  :cognitect.aws.dynamodb/WriteRequest
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.WriteRequest/DeleteRequest
     :cognitect.aws.dynamodb.WriteRequest/PutRequest]))

(s/def
  :cognitect.aws.dynamodb/TransactWriteItemsInput
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.TransactWriteItemsInput/TransactItems]
    :opt-un
    [:cognitect.aws.dynamodb.TransactWriteItemsInput/ClientRequestToken
     :cognitect.aws.dynamodb.TransactWriteItemsInput/ReturnConsumedCapacity
     :cognitect.aws.dynamodb.TransactWriteItemsInput/ReturnItemCollectionMetrics]))

(s/def
  :cognitect.aws.dynamodb/ExecuteStatementOutput
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.ExecuteStatementOutput/LastEvaluatedKey
     :cognitect.aws.dynamodb.ExecuteStatementOutput/NextToken
     :cognitect.aws.dynamodb.ExecuteStatementOutput/ConsumedCapacity
     :cognitect.aws.dynamodb.ExecuteStatementOutput/Items]))

(s/def
  :cognitect.aws.dynamodb/ReplicaGlobalSecondaryIndexSettingsUpdateList
  (s/coll-of
    :cognitect.aws.dynamodb/ReplicaGlobalSecondaryIndexSettingsUpdate
    :min-count
    1
    :max-count
    20))

(s/def
  :cognitect.aws.dynamodb/ListAttributeValue
  (s/coll-of :cognitect.aws.dynamodb/AttributeValue))

(s/def
  :cognitect.aws.dynamodb/ImportTableInput
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.ImportTableInput/S3BucketSource
     :cognitect.aws.dynamodb.ImportTableInput/InputFormat
     :cognitect.aws.dynamodb.ImportTableInput/TableCreationParameters]
    :opt-un
    [:cognitect.aws.dynamodb.ImportTableInput/InputFormatOptions
     :cognitect.aws.dynamodb.ImportTableInput/InputCompressionType
     :cognitect.aws.dynamodb.ImportTableInput/ClientToken]))

(s/def :cognitect.aws.dynamodb/ScalarAttributeType (s/spec string? :gen #(s/gen #{"S" "B" "N"})))

(s/def
  :cognitect.aws.dynamodb/ListImportsInput
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.ListImportsInput/PageSize
     :cognitect.aws.dynamodb.ListImportsInput/NextToken
     :cognitect.aws.dynamodb.ListImportsInput/TableArn]))

(s/def :cognitect.aws.dynamodb/StringAttributeValue string?)

(s/def
  :cognitect.aws.dynamodb/UpdateReplicationGroupMemberAction
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.UpdateReplicationGroupMemberAction/RegionName]
    :opt-un
    [:cognitect.aws.dynamodb.UpdateReplicationGroupMemberAction/ProvisionedThroughputOverride
     :cognitect.aws.dynamodb.UpdateReplicationGroupMemberAction/KMSMasterKeyId
     :cognitect.aws.dynamodb.UpdateReplicationGroupMemberAction/TableClassOverride
     :cognitect.aws.dynamodb.UpdateReplicationGroupMemberAction/GlobalSecondaryIndexes]))

(s/def
  :cognitect.aws.dynamodb/ItemCount
  (s/spec (s/and int? #(<= 0 %)) :gen #(gen/choose 0 Long/MAX_VALUE)))

(s/def
  :cognitect.aws.dynamodb/LocalSecondaryIndexInfo
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.LocalSecondaryIndexInfo/Projection
     :cognitect.aws.dynamodb.LocalSecondaryIndexInfo/IndexName
     :cognitect.aws.dynamodb.LocalSecondaryIndexInfo/KeySchema]))

(s/def
  :cognitect.aws.dynamodb/ItemCollectionKeyAttributeMap
  (s/map-of :cognitect.aws.dynamodb/AttributeName :cognitect.aws.dynamodb/AttributeValue))

(s/def
  :cognitect.aws.dynamodb/ReplicationGroupUpdate
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.ReplicationGroupUpdate/Delete
     :cognitect.aws.dynamodb.ReplicationGroupUpdate/Update
     :cognitect.aws.dynamodb.ReplicationGroupUpdate/Create]))

(s/def :cognitect.aws.dynamodb/ExportFormat (s/spec string? :gen #(s/gen #{"ION" "DYNAMODB_JSON"})))

(s/def
  :cognitect.aws.dynamodb/S3Bucket
  (s/spec #(re-matches (re-pattern "^[a-z0-9A-Z]+[\\.\\-\\w]*[a-z0-9A-Z]+$") %) :gen #(gen/string)))

(s/def :cognitect.aws.dynamodb/DescribeLimitsInput (s/keys))

(s/def
  :cognitect.aws.dynamodb/UpdateTableOutput
  (s/keys :opt-un [:cognitect.aws.dynamodb.UpdateTableOutput/TableDescription]))

(s/def
  :cognitect.aws.dynamodb/PutItemInput
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.PutItemInput/TableName :cognitect.aws.dynamodb.PutItemInput/Item]
    :opt-un
    [:cognitect.aws.dynamodb.PutItemInput/Expected
     :cognitect.aws.dynamodb.PutItemInput/ReturnValuesOnConditionCheckFailure
     :cognitect.aws.dynamodb.PutItemInput/ReturnConsumedCapacity
     :cognitect.aws.dynamodb.PutItemInput/ConditionalOperator
     :cognitect.aws.dynamodb.PutItemInput/ConditionExpression
     :cognitect.aws.dynamodb.PutItemInput/ExpressionAttributeNames
     :cognitect.aws.dynamodb.PutItemInput/ReturnValues
     :cognitect.aws.dynamodb.PutItemInput/ExpressionAttributeValues
     :cognitect.aws.dynamodb.PutItemInput/ReturnItemCollectionMetrics]))

(s/def
  :cognitect.aws.dynamodb/DeleteItemInput
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.DeleteItemInput/TableName :cognitect.aws.dynamodb.DeleteItemInput/Key]
    :opt-un
    [:cognitect.aws.dynamodb.DeleteItemInput/Expected
     :cognitect.aws.dynamodb.DeleteItemInput/ReturnValuesOnConditionCheckFailure
     :cognitect.aws.dynamodb.DeleteItemInput/ReturnConsumedCapacity
     :cognitect.aws.dynamodb.DeleteItemInput/ConditionalOperator
     :cognitect.aws.dynamodb.DeleteItemInput/ConditionExpression
     :cognitect.aws.dynamodb.DeleteItemInput/ExpressionAttributeNames
     :cognitect.aws.dynamodb.DeleteItemInput/ReturnValues
     :cognitect.aws.dynamodb.DeleteItemInput/ExpressionAttributeValues
     :cognitect.aws.dynamodb.DeleteItemInput/ReturnItemCollectionMetrics]))

(s/def :cognitect.aws.dynamodb/TableNameList (s/coll-of :cognitect.aws.dynamodb/TableName))

(s/def :cognitect.aws.dynamodb/S3SseAlgorithm (s/spec string? :gen #(s/gen #{"KMS" "AES256"})))

(s/def
  :cognitect.aws.dynamodb/UpdateGlobalTableOutput
  (s/keys :opt-un [:cognitect.aws.dynamodb.UpdateGlobalTableOutput/GlobalTableDescription]))

(s/def
  :cognitect.aws.dynamodb/ContinuousBackupsStatus
  (s/spec string? :gen #(s/gen #{"DISABLED" "ENABLED"})))

(s/def
  :cognitect.aws.dynamodb/BatchStatementErrorCodeEnum
  (s/spec
    string?
    :gen
    #(s/gen
      #{"ThrottlingError"
        "ValidationError"
        "AccessDenied"
        "ProvisionedThroughputExceeded"
        "TransactionConflict"
        "RequestLimitExceeded"
        "ConditionalCheckFailed"
        "DuplicateItem"
        "InternalServerError"
        "ResourceNotFound"
        "ItemCollectionSizeLimitExceeded"})))

(s/def
  :cognitect.aws.dynamodb/UpdateGlobalSecondaryIndexAction
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.UpdateGlobalSecondaryIndexAction/IndexName
     :cognitect.aws.dynamodb.UpdateGlobalSecondaryIndexAction/ProvisionedThroughput]))

(s/def :cognitect.aws.dynamodb/ExportTime inst?)

(s/def
  :cognitect.aws.dynamodb/GlobalTableGlobalSecondaryIndexSettingsUpdate
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.GlobalTableGlobalSecondaryIndexSettingsUpdate/IndexName]
    :opt-un
    [:cognitect.aws.dynamodb.GlobalTableGlobalSecondaryIndexSettingsUpdate/ProvisionedWriteCapacityUnits
     :cognitect.aws.dynamodb.GlobalTableGlobalSecondaryIndexSettingsUpdate/ProvisionedWriteCapacityAutoScalingSettingsUpdate]))

(s/def :cognitect.aws.dynamodb/ReplicaList (s/coll-of :cognitect.aws.dynamodb/Replica))

(s/def
  :cognitect.aws.dynamodb/AutoScalingPolicyUpdate
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.AutoScalingPolicyUpdate/TargetTrackingScalingPolicyConfiguration]
    :opt-un
    [:cognitect.aws.dynamodb.AutoScalingPolicyUpdate/PolicyName]))

(s/def
  :cognitect.aws.dynamodb/Integer
  (s/spec int? :gen #(gen/choose Long/MIN_VALUE Long/MAX_VALUE)))

(s/def
  :cognitect.aws.dynamodb/ReplicaStatus
  (s/spec
    string?
    :gen
    #(s/gen
      #{"UPDATING"
        "REGION_DISABLED"
        "DELETING"
        "CREATING"
        "ACTIVE"
        "INACCESSIBLE_ENCRYPTION_CREDENTIALS"
        "CREATION_FAILED"})))

(s/def
  :cognitect.aws.dynamodb/ScanInput
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.ScanInput/TableName]
    :opt-un
    [:cognitect.aws.dynamodb.ScanInput/ScanFilter
     :cognitect.aws.dynamodb.ScanInput/IndexName
     :cognitect.aws.dynamodb.ScanInput/AttributesToGet
     :cognitect.aws.dynamodb.ScanInput/Limit
     :cognitect.aws.dynamodb.ScanInput/ConsistentRead
     :cognitect.aws.dynamodb.ScanInput/ReturnConsumedCapacity
     :cognitect.aws.dynamodb.ScanInput/ConditionalOperator
     :cognitect.aws.dynamodb.ScanInput/FilterExpression
     :cognitect.aws.dynamodb.ScanInput/Segment
     :cognitect.aws.dynamodb.ScanInput/ExclusiveStartKey
     :cognitect.aws.dynamodb.ScanInput/TotalSegments
     :cognitect.aws.dynamodb.ScanInput/ExpressionAttributeNames
     :cognitect.aws.dynamodb.ScanInput/ProjectionExpression
     :cognitect.aws.dynamodb.ScanInput/Select
     :cognitect.aws.dynamodb.ScanInput/ExpressionAttributeValues]))

(s/def
  :cognitect.aws.dynamodb/DescribeImportInput
  (s/keys :req-un [:cognitect.aws.dynamodb.DescribeImportInput/ImportArn]))

(s/def
  :cognitect.aws.dynamodb/ConsumedCapacityMultiple
  (s/coll-of :cognitect.aws.dynamodb/ConsumedCapacity))

(s/def :cognitect.aws.dynamodb/ImportEndTime inst?)

(s/def
  :cognitect.aws.dynamodb/CreateBackupInput
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.CreateBackupInput/TableName
     :cognitect.aws.dynamodb.CreateBackupInput/BackupName]))

(s/def
  :cognitect.aws.dynamodb/ExportStatus
  (s/spec string? :gen #(s/gen #{"COMPLETED" "IN_PROGRESS" "FAILED"})))

(s/def
  :cognitect.aws.dynamodb/CreateReplicationGroupMemberAction
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.CreateReplicationGroupMemberAction/RegionName]
    :opt-un
    [:cognitect.aws.dynamodb.CreateReplicationGroupMemberAction/ProvisionedThroughputOverride
     :cognitect.aws.dynamodb.CreateReplicationGroupMemberAction/KMSMasterKeyId
     :cognitect.aws.dynamodb.CreateReplicationGroupMemberAction/TableClassOverride
     :cognitect.aws.dynamodb.CreateReplicationGroupMemberAction/GlobalSecondaryIndexes]))

(s/def
  :cognitect.aws.dynamodb/LocalSecondaryIndexList
  (s/coll-of :cognitect.aws.dynamodb/LocalSecondaryIndex))

(s/def
  :cognitect.aws.dynamodb/DescribeKinesisStreamingDestinationInput
  (s/keys :req-un [:cognitect.aws.dynamodb.DescribeKinesisStreamingDestinationInput/TableName]))

(s/def
  :cognitect.aws.dynamodb/GlobalSecondaryIndexList
  (s/coll-of :cognitect.aws.dynamodb/GlobalSecondaryIndex))

(s/def :cognitect.aws.dynamodb/ConditionExpression string?)

(s/def
  :cognitect.aws.dynamodb/ClientRequestToken
  (s/spec
    (s/and string? #(<= 1 (count %) 36))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 1 36) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.dynamodb/WriteRequests
  (s/coll-of :cognitect.aws.dynamodb/WriteRequest :min-count 1 :max-count 25))

(s/def
  :cognitect.aws.dynamodb/BackupDescription
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.BackupDescription/SourceTableFeatureDetails
     :cognitect.aws.dynamodb.BackupDescription/SourceTableDetails
     :cognitect.aws.dynamodb.BackupDescription/BackupDetails]))

(s/def
  :cognitect.aws.dynamodb/StreamSpecification
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.StreamSpecification/StreamEnabled]
    :opt-un
    [:cognitect.aws.dynamodb.StreamSpecification/StreamViewType]))

(s/def
  :cognitect.aws.dynamodb/DeleteTableOutput
  (s/keys :opt-un [:cognitect.aws.dynamodb.DeleteTableOutput/TableDescription]))

(s/def
  :cognitect.aws.dynamodb/UpdateTableReplicaAutoScalingOutput
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.UpdateTableReplicaAutoScalingOutput/TableAutoScalingDescription]))

(s/def
  :cognitect.aws.dynamodb/TableId
  (s/spec
    #(re-matches (re-pattern "[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}") %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.dynamodb/InputFormatOptions
  (s/keys :opt-un [:cognitect.aws.dynamodb.InputFormatOptions/Csv]))

(s/def :cognitect.aws.dynamodb/KeyExpression string?)

(s/def
  :cognitect.aws.dynamodb/RestoreTableFromBackupOutput
  (s/keys :opt-un [:cognitect.aws.dynamodb.RestoreTableFromBackupOutput/TableDescription]))

(s/def
  :cognitect.aws.dynamodb/GlobalSecondaryIndexes
  (s/coll-of :cognitect.aws.dynamodb/GlobalSecondaryIndexInfo))

(s/def
  :cognitect.aws.dynamodb/GlobalSecondaryIndexDescription
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.GlobalSecondaryIndexDescription/Backfilling
     :cognitect.aws.dynamodb.GlobalSecondaryIndexDescription/IndexStatus
     :cognitect.aws.dynamodb.GlobalSecondaryIndexDescription/Projection
     :cognitect.aws.dynamodb.GlobalSecondaryIndexDescription/IndexName
     :cognitect.aws.dynamodb.GlobalSecondaryIndexDescription/IndexSizeBytes
     :cognitect.aws.dynamodb.GlobalSecondaryIndexDescription/IndexArn
     :cognitect.aws.dynamodb.GlobalSecondaryIndexDescription/KeySchema
     :cognitect.aws.dynamodb.GlobalSecondaryIndexDescription/ItemCount
     :cognitect.aws.dynamodb.GlobalSecondaryIndexDescription/ProvisionedThroughput]))

(s/def
  :cognitect.aws.dynamodb/DeleteBackupOutput
  (s/keys :opt-un [:cognitect.aws.dynamodb.DeleteBackupOutput/BackupDescription]))

(s/def
  :cognitect.aws.dynamodb/DescribeLimitsOutput
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.DescribeLimitsOutput/AccountMaxReadCapacityUnits
     :cognitect.aws.dynamodb.DescribeLimitsOutput/TableMaxWriteCapacityUnits
     :cognitect.aws.dynamodb.DescribeLimitsOutput/AccountMaxWriteCapacityUnits
     :cognitect.aws.dynamodb.DescribeLimitsOutput/TableMaxReadCapacityUnits]))

(s/def :cognitect.aws.dynamodb/ImportStartTime inst?)

(s/def :cognitect.aws.dynamodb/ProjectionExpression string?)

(s/def
  :cognitect.aws.dynamodb/StringSetAttributeValue
  (s/coll-of :cognitect.aws.dynamodb/StringAttributeValue))

(s/def
  :cognitect.aws.dynamodb/LocalSecondaryIndexDescription
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.LocalSecondaryIndexDescription/Projection
     :cognitect.aws.dynamodb.LocalSecondaryIndexDescription/IndexName
     :cognitect.aws.dynamodb.LocalSecondaryIndexDescription/IndexSizeBytes
     :cognitect.aws.dynamodb.LocalSecondaryIndexDescription/IndexArn
     :cognitect.aws.dynamodb.LocalSecondaryIndexDescription/KeySchema
     :cognitect.aws.dynamodb.LocalSecondaryIndexDescription/ItemCount]))

(s/def
  :cognitect.aws.dynamodb/ConsumedCapacityUnits
  (s/spec double? :gen #(gen/double* {:infinite? false, :NaN? false})))

(s/def
  :cognitect.aws.dynamodb/TransactWriteItem
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.TransactWriteItem/ConditionCheck
     :cognitect.aws.dynamodb.TransactWriteItem/Put
     :cognitect.aws.dynamodb.TransactWriteItem/Delete
     :cognitect.aws.dynamodb.TransactWriteItem/Update]))

(s/def :cognitect.aws.dynamodb/GlobalTableArnString string?)

(s/def :cognitect.aws.dynamodb/TableCreationDateTime inst?)

(s/def
  :cognitect.aws.dynamodb/ReplicaGlobalSecondaryIndexSettingsDescriptionList
  (s/coll-of :cognitect.aws.dynamodb/ReplicaGlobalSecondaryIndexSettingsDescription))

(s/def
  :cognitect.aws.dynamodb/SourceTableFeatureDetails
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.SourceTableFeatureDetails/TimeToLiveDescription
     :cognitect.aws.dynamodb.SourceTableFeatureDetails/StreamDescription
     :cognitect.aws.dynamodb.SourceTableFeatureDetails/LocalSecondaryIndexes
     :cognitect.aws.dynamodb.SourceTableFeatureDetails/GlobalSecondaryIndexes
     :cognitect.aws.dynamodb.SourceTableFeatureDetails/SSEDescription]))

(s/def :cognitect.aws.dynamodb/TimeToLiveEnabled boolean?)

(s/def :cognitect.aws.dynamodb/KMSMasterKeyArn string?)

(s/def
  :cognitect.aws.dynamodb/TransactGetItemsOutput
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.TransactGetItemsOutput/Responses
     :cognitect.aws.dynamodb.TransactGetItemsOutput/ConsumedCapacity]))

(s/def
  :cognitect.aws.dynamodb/ItemCollectionSizeEstimateBound
  (s/spec double? :gen #(gen/double* {:infinite? false, :NaN? false})))

(s/def
  :cognitect.aws.dynamodb/MapAttributeValue
  (s/map-of :cognitect.aws.dynamodb/AttributeName :cognitect.aws.dynamodb/AttributeValue))

(s/def
  :cognitect.aws.dynamodb/PartiQLNextToken
  (s/spec
    (s/and string? #(<= 1 (count %) 32768))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 1 32768) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.dynamodb/ExecuteStatementInput
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.ExecuteStatementInput/Statement]
    :opt-un
    [:cognitect.aws.dynamodb.ExecuteStatementInput/Parameters
     :cognitect.aws.dynamodb.ExecuteStatementInput/Limit
     :cognitect.aws.dynamodb.ExecuteStatementInput/NextToken
     :cognitect.aws.dynamodb.ExecuteStatementInput/ConsistentRead
     :cognitect.aws.dynamodb.ExecuteStatementInput/ReturnValuesOnConditionCheckFailure
     :cognitect.aws.dynamodb.ExecuteStatementInput/ReturnConsumedCapacity]))

(s/def :cognitect.aws.dynamodb/ArchivalReason string?)

(s/def
  :cognitect.aws.dynamodb/StreamArn
  (s/spec
    (s/and string? #(<= 37 (count %) 1024))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 37 1024) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.dynamodb/Put
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.Put/Item :cognitect.aws.dynamodb.Put/TableName]
    :opt-un
    [:cognitect.aws.dynamodb.Put/ReturnValuesOnConditionCheckFailure
     :cognitect.aws.dynamodb.Put/ConditionExpression
     :cognitect.aws.dynamodb.Put/ExpressionAttributeNames
     :cognitect.aws.dynamodb.Put/ExpressionAttributeValues]))

(s/def
  :cognitect.aws.dynamodb/ListTablesInput
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.ListTablesInput/ExclusiveStartTableName
     :cognitect.aws.dynamodb.ListTablesInput/Limit]))

(s/def
  :cognitect.aws.dynamodb/TransactGetItemsInput
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.TransactGetItemsInput/TransactItems]
    :opt-un
    [:cognitect.aws.dynamodb.TransactGetItemsInput/ReturnConsumedCapacity]))

(s/def
  :cognitect.aws.dynamodb/AutoScalingPolicyDescription
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.AutoScalingPolicyDescription/PolicyName
     :cognitect.aws.dynamodb.AutoScalingPolicyDescription/TargetTrackingScalingPolicyConfiguration]))

(s/def
  :cognitect.aws.dynamodb/UpdateTableInput
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.UpdateTableInput/TableName]
    :opt-un
    [:cognitect.aws.dynamodb.UpdateTableInput/StreamSpecification
     :cognitect.aws.dynamodb.UpdateTableInput/AttributeDefinitions
     :cognitect.aws.dynamodb.UpdateTableInput/TableClass
     :cognitect.aws.dynamodb.UpdateTableInput/GlobalSecondaryIndexUpdates
     :cognitect.aws.dynamodb.UpdateTableInput/SSESpecification
     :cognitect.aws.dynamodb.UpdateTableInput/BillingMode
     :cognitect.aws.dynamodb.UpdateTableInput/DeletionProtectionEnabled
     :cognitect.aws.dynamodb.UpdateTableInput/ProvisionedThroughput
     :cognitect.aws.dynamodb.UpdateTableInput/ReplicaUpdates]))

(s/def :cognitect.aws.dynamodb/KMSMasterKeyId string?)

(s/def
  :cognitect.aws.dynamodb/BackupType
  (s/spec string? :gen #(s/gen #{"SYSTEM" "USER" "AWS_BACKUP"})))

(s/def
  :cognitect.aws.dynamodb/PartiQLBatchRequest
  (s/coll-of :cognitect.aws.dynamodb/BatchStatementRequest :min-count 1 :max-count 25))

(s/def
  :cognitect.aws.dynamodb/CreateReplicaAction
  (s/keys :req-un [:cognitect.aws.dynamodb.CreateReplicaAction/RegionName]))

(s/def
  :cognitect.aws.dynamodb/KinesisStreamingDestinationOutput
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.KinesisStreamingDestinationOutput/StreamArn
     :cognitect.aws.dynamodb.KinesisStreamingDestinationOutput/DestinationStatus
     :cognitect.aws.dynamodb.KinesisStreamingDestinationOutput/TableName]))

(s/def
  :cognitect.aws.dynamodb/DeleteRequest
  (s/keys :req-un [:cognitect.aws.dynamodb.DeleteRequest/Key]))

(s/def :cognitect.aws.dynamodb/RestoreInProgress boolean?)

(s/def
  :cognitect.aws.dynamodb/ReplicaAutoScalingUpdateList
  (s/coll-of :cognitect.aws.dynamodb/ReplicaAutoScalingUpdate :min-count 1))

(s/def
  :cognitect.aws.dynamodb/BatchStatementError
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.BatchStatementError/Message
     :cognitect.aws.dynamodb.BatchStatementError/Code
     :cognitect.aws.dynamodb.BatchStatementError/Item]))

(s/def
  :cognitect.aws.dynamodb/UntagResourceInput
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.UntagResourceInput/ResourceArn
     :cognitect.aws.dynamodb.UntagResourceInput/TagKeys]))

(s/def
  :cognitect.aws.dynamodb/TransactWriteItemsOutput
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.TransactWriteItemsOutput/ItemCollectionMetrics
     :cognitect.aws.dynamodb.TransactWriteItemsOutput/ConsumedCapacity]))

(s/def
  :cognitect.aws.dynamodb/NonNegativeLongObject
  (s/spec (s/and int? #(<= 0 %)) :gen #(gen/choose 0 Long/MAX_VALUE)))

(s/def :cognitect.aws.dynamodb/FailureCode string?)

(s/def
  :cognitect.aws.dynamodb/BackupDetails
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.BackupDetails/BackupArn
     :cognitect.aws.dynamodb.BackupDetails/BackupName
     :cognitect.aws.dynamodb.BackupDetails/BackupStatus
     :cognitect.aws.dynamodb.BackupDetails/BackupType
     :cognitect.aws.dynamodb.BackupDetails/BackupCreationDateTime]
    :opt-un
    [:cognitect.aws.dynamodb.BackupDetails/BackupExpiryDateTime
     :cognitect.aws.dynamodb.BackupDetails/BackupSizeBytes]))

(s/def
  :cognitect.aws.dynamodb/RestoreTableToPointInTimeInput
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.RestoreTableToPointInTimeInput/TargetTableName]
    :opt-un
    [:cognitect.aws.dynamodb.RestoreTableToPointInTimeInput/BillingModeOverride
     :cognitect.aws.dynamodb.RestoreTableToPointInTimeInput/ProvisionedThroughputOverride
     :cognitect.aws.dynamodb.RestoreTableToPointInTimeInput/RestoreDateTime
     :cognitect.aws.dynamodb.RestoreTableToPointInTimeInput/SourceTableArn
     :cognitect.aws.dynamodb.RestoreTableToPointInTimeInput/LocalSecondaryIndexOverride
     :cognitect.aws.dynamodb.RestoreTableToPointInTimeInput/SourceTableName
     :cognitect.aws.dynamodb.RestoreTableToPointInTimeInput/GlobalSecondaryIndexOverride
     :cognitect.aws.dynamodb.RestoreTableToPointInTimeInput/UseLatestRestorableTime
     :cognitect.aws.dynamodb.RestoreTableToPointInTimeInput/SSESpecificationOverride]))

(s/def
  :cognitect.aws.dynamodb/KeySchema
  (s/coll-of :cognitect.aws.dynamodb/KeySchemaElement :min-count 1 :max-count 2))

(s/def
  :cognitect.aws.dynamodb/ProvisionedThroughput
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.ProvisionedThroughput/ReadCapacityUnits
     :cognitect.aws.dynamodb.ProvisionedThroughput/WriteCapacityUnits]))

(s/def
  :cognitect.aws.dynamodb/DescribeTimeToLiveOutput
  (s/keys :opt-un [:cognitect.aws.dynamodb.DescribeTimeToLiveOutput/TimeToLiveDescription]))

(s/def
  :cognitect.aws.dynamodb/ReplicaGlobalSecondaryIndexAutoScalingDescriptionList
  (s/coll-of :cognitect.aws.dynamodb/ReplicaGlobalSecondaryIndexAutoScalingDescription))

(s/def
  :cognitect.aws.dynamodb/Delete
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.Delete/Key :cognitect.aws.dynamodb.Delete/TableName]
    :opt-un
    [:cognitect.aws.dynamodb.Delete/ReturnValuesOnConditionCheckFailure
     :cognitect.aws.dynamodb.Delete/ConditionExpression
     :cognitect.aws.dynamodb.Delete/ExpressionAttributeNames
     :cognitect.aws.dynamodb.Delete/ExpressionAttributeValues]))

(s/def
  :cognitect.aws.dynamodb/SourceTableDetails
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.SourceTableDetails/TableName
     :cognitect.aws.dynamodb.SourceTableDetails/TableId
     :cognitect.aws.dynamodb.SourceTableDetails/KeySchema
     :cognitect.aws.dynamodb.SourceTableDetails/TableCreationDateTime
     :cognitect.aws.dynamodb.SourceTableDetails/ProvisionedThroughput]
    :opt-un
    [:cognitect.aws.dynamodb.SourceTableDetails/TableSizeBytes
     :cognitect.aws.dynamodb.SourceTableDetails/BillingMode
     :cognitect.aws.dynamodb.SourceTableDetails/ItemCount
     :cognitect.aws.dynamodb.SourceTableDetails/TableArn]))

(s/def
  :cognitect.aws.dynamodb/DescribeEndpointsResponse
  (s/keys :req-un [:cognitect.aws.dynamodb.DescribeEndpointsResponse/Endpoints]))

(s/def
  :cognitect.aws.dynamodb/IntegerObject
  (s/spec int? :gen #(gen/choose Long/MIN_VALUE Long/MAX_VALUE)))

(s/def
  :cognitect.aws.dynamodb/PreparedStatementParameters
  (s/coll-of :cognitect.aws.dynamodb/AttributeValue :min-count 1))

(s/def
  :cognitect.aws.dynamodb/CreateGlobalSecondaryIndexAction
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.CreateGlobalSecondaryIndexAction/IndexName
     :cognitect.aws.dynamodb.CreateGlobalSecondaryIndexAction/KeySchema
     :cognitect.aws.dynamodb.CreateGlobalSecondaryIndexAction/Projection]
    :opt-un
    [:cognitect.aws.dynamodb.CreateGlobalSecondaryIndexAction/ProvisionedThroughput]))

(s/def
  :cognitect.aws.dynamodb/AttributeValueUpdate
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.AttributeValueUpdate/Value
     :cognitect.aws.dynamodb.AttributeValueUpdate/Action]))

(s/def
  :cognitect.aws.dynamodb/ExecuteTransactionOutput
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.ExecuteTransactionOutput/Responses
     :cognitect.aws.dynamodb.ExecuteTransactionOutput/ConsumedCapacity]))

(s/def
  :cognitect.aws.dynamodb/LocalSecondaryIndexes
  (s/coll-of :cognitect.aws.dynamodb/LocalSecondaryIndexInfo))

(s/def
  :cognitect.aws.dynamodb/SecondaryIndexesCapacityMap
  (s/map-of :cognitect.aws.dynamodb/IndexName :cognitect.aws.dynamodb/Capacity))

(s/def
  :cognitect.aws.dynamodb/TableDescription
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.TableDescription/StreamSpecification
     :cognitect.aws.dynamodb.TableDescription/LatestStreamArn
     :cognitect.aws.dynamodb.TableDescription/Replicas
     :cognitect.aws.dynamodb.TableDescription/TableId
     :cognitect.aws.dynamodb.TableDescription/AttributeDefinitions
     :cognitect.aws.dynamodb.TableDescription/KeySchema
     :cognitect.aws.dynamodb.TableDescription/CreationDateTime
     :cognitect.aws.dynamodb.TableDescription/LocalSecondaryIndexes
     :cognitect.aws.dynamodb.TableDescription/ArchivalSummary
     :cognitect.aws.dynamodb.TableDescription/TableStatus
     :cognitect.aws.dynamodb.TableDescription/TableSizeBytes
     :cognitect.aws.dynamodb.TableDescription/ItemCount
     :cognitect.aws.dynamodb.TableDescription/TableName
     :cognitect.aws.dynamodb.TableDescription/GlobalSecondaryIndexes
     :cognitect.aws.dynamodb.TableDescription/TableArn
     :cognitect.aws.dynamodb.TableDescription/RestoreSummary
     :cognitect.aws.dynamodb.TableDescription/LatestStreamLabel
     :cognitect.aws.dynamodb.TableDescription/DeletionProtectionEnabled
     :cognitect.aws.dynamodb.TableDescription/TableClassSummary
     :cognitect.aws.dynamodb.TableDescription/ProvisionedThroughput
     :cognitect.aws.dynamodb.TableDescription/SSEDescription
     :cognitect.aws.dynamodb.TableDescription/BillingModeSummary
     :cognitect.aws.dynamodb.TableDescription/GlobalTableVersion]))

(s/def
  :cognitect.aws.dynamodb/PointInTimeRecoverySpecification
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.PointInTimeRecoverySpecification/PointInTimeRecoveryEnabled]))

(s/def
  :cognitect.aws.dynamodb/S3SseKmsKeyId
  (s/spec
    (s/and string? #(<= 1 (count %) 2048))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 1 2048) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.dynamodb/ContributorInsightsStatus
  (s/spec string? :gen #(s/gen #{"DISABLED" "DISABLING" "ENABLING" "FAILED" "ENABLED"})))

(s/def :cognitect.aws.dynamodb/UpdateExpression string?)

(s/def :cognitect.aws.dynamodb/ExportNextToken string?)

(s/def
  :cognitect.aws.dynamodb/ReplicaGlobalSecondaryIndexDescriptionList
  (s/coll-of :cognitect.aws.dynamodb/ReplicaGlobalSecondaryIndexDescription))

(s/def
  :cognitect.aws.dynamodb/ArchivalSummary
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.ArchivalSummary/ArchivalDateTime
     :cognitect.aws.dynamodb.ArchivalSummary/ArchivalReason
     :cognitect.aws.dynamodb.ArchivalSummary/ArchivalBackupArn]))

(s/def
  :cognitect.aws.dynamodb/PositiveLongObject
  (s/spec (s/and int? #(<= 1 %)) :gen #(gen/choose 1 Long/MAX_VALUE)))

(s/def
  :cognitect.aws.dynamodb/Get
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.Get/Key :cognitect.aws.dynamodb.Get/TableName]
    :opt-un
    [:cognitect.aws.dynamodb.Get/ExpressionAttributeNames
     :cognitect.aws.dynamodb.Get/ProjectionExpression]))

(s/def :cognitect.aws.dynamodb/Date inst?)

(s/def :cognitect.aws.dynamodb/ExceptionDescription string?)

(s/def :cognitect.aws.dynamodb/SSEType (s/spec string? :gen #(s/gen #{"KMS" "AES256"})))

(s/def
  :cognitect.aws.dynamodb/UpdateGlobalTableInput
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.UpdateGlobalTableInput/GlobalTableName
     :cognitect.aws.dynamodb.UpdateGlobalTableInput/ReplicaUpdates]))

(s/def
  :cognitect.aws.dynamodb/AutoScalingPolicyName
  (s/spec #(re-matches (re-pattern "\\p{Print}+") %) :gen #(gen/string)))

(s/def :cognitect.aws.dynamodb/BackupCreationDateTime inst?)

(s/def
  :cognitect.aws.dynamodb/AutoScalingTargetTrackingScalingPolicyConfigurationUpdate
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.AutoScalingTargetTrackingScalingPolicyConfigurationUpdate/TargetValue]
    :opt-un
    [:cognitect.aws.dynamodb.AutoScalingTargetTrackingScalingPolicyConfigurationUpdate/ScaleInCooldown
     :cognitect.aws.dynamodb.AutoScalingTargetTrackingScalingPolicyConfigurationUpdate/DisableScaleIn
     :cognitect.aws.dynamodb.AutoScalingTargetTrackingScalingPolicyConfigurationUpdate/ScaleOutCooldown]))

(s/def :cognitect.aws.dynamodb/TimeRangeLowerBound inst?)

(s/def
  :cognitect.aws.dynamodb/UpdateGlobalTableSettingsOutput
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.UpdateGlobalTableSettingsOutput/GlobalTableName
     :cognitect.aws.dynamodb.UpdateGlobalTableSettingsOutput/ReplicaSettings]))

(s/def :cognitect.aws.dynamodb/ConsistentRead boolean?)

(s/def
  :cognitect.aws.dynamodb/PutItemOutput
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.PutItemOutput/ItemCollectionMetrics
     :cognitect.aws.dynamodb.PutItemOutput/Attributes
     :cognitect.aws.dynamodb.PutItemOutput/ConsumedCapacity]))

(s/def
  :cognitect.aws.dynamodb/ResourceArnString
  (s/spec
    (s/and string? #(<= 1 (count %) 1283))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 1 1283) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.dynamodb/LongObject
  (s/spec int? :gen #(gen/choose Long/MIN_VALUE Long/MAX_VALUE)))

(s/def :cognitect.aws.dynamodb/Long (s/spec int? :gen #(gen/choose Long/MIN_VALUE Long/MAX_VALUE)))

(s/def
  :cognitect.aws.dynamodb/ErrorCount
  (s/spec (s/and int? #(<= 0 %)) :gen #(gen/choose 0 Long/MAX_VALUE)))

(s/def
  :cognitect.aws.dynamodb/CreateTableInput
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.CreateTableInput/AttributeDefinitions
     :cognitect.aws.dynamodb.CreateTableInput/TableName
     :cognitect.aws.dynamodb.CreateTableInput/KeySchema]
    :opt-un
    [:cognitect.aws.dynamodb.CreateTableInput/StreamSpecification
     :cognitect.aws.dynamodb.CreateTableInput/Tags
     :cognitect.aws.dynamodb.CreateTableInput/TableClass
     :cognitect.aws.dynamodb.CreateTableInput/LocalSecondaryIndexes
     :cognitect.aws.dynamodb.CreateTableInput/SSESpecification
     :cognitect.aws.dynamodb.CreateTableInput/BillingMode
     :cognitect.aws.dynamodb.CreateTableInput/GlobalSecondaryIndexes
     :cognitect.aws.dynamodb.CreateTableInput/DeletionProtectionEnabled
     :cognitect.aws.dynamodb.CreateTableInput/ProvisionedThroughput]))

(s/def
  :cognitect.aws.dynamodb/ConditionCheck
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.ConditionCheck/Key
     :cognitect.aws.dynamodb.ConditionCheck/TableName
     :cognitect.aws.dynamodb.ConditionCheck/ConditionExpression]
    :opt-un
    [:cognitect.aws.dynamodb.ConditionCheck/ReturnValuesOnConditionCheckFailure
     :cognitect.aws.dynamodb.ConditionCheck/ExpressionAttributeNames
     :cognitect.aws.dynamodb.ConditionCheck/ExpressionAttributeValues]))

(s/def
  :cognitect.aws.dynamodb/DescribeContributorInsightsInput
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.DescribeContributorInsightsInput/TableName]
    :opt-un
    [:cognitect.aws.dynamodb.DescribeContributorInsightsInput/IndexName]))

(s/def
  :cognitect.aws.dynamodb/ListTablesInputLimit
  (s/spec (s/and int? #(<= 1 % 100)) :gen #(gen/choose 1 100)))

(s/def
  :cognitect.aws.dynamodb/BatchStatementRequest
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.BatchStatementRequest/Statement]
    :opt-un
    [:cognitect.aws.dynamodb.BatchStatementRequest/Parameters
     :cognitect.aws.dynamodb.BatchStatementRequest/ConsistentRead
     :cognitect.aws.dynamodb.BatchStatementRequest/ReturnValuesOnConditionCheckFailure]))

(s/def
  :cognitect.aws.dynamodb/PointInTimeRecoveryDescription
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.PointInTimeRecoveryDescription/EarliestRestorableDateTime
     :cognitect.aws.dynamodb.PointInTimeRecoveryDescription/PointInTimeRecoveryStatus
     :cognitect.aws.dynamodb.PointInTimeRecoveryDescription/LatestRestorableDateTime]))

(s/def
  :cognitect.aws.dynamodb/BackupName
  (s/spec #(re-matches (re-pattern "[a-zA-Z0-9_.-]+") %) :gen #(gen/string)))

(s/def :cognitect.aws.dynamodb/LastUpdateDateTime inst?)

(s/def
  :cognitect.aws.dynamodb/ReplicaAutoScalingDescriptionList
  (s/coll-of :cognitect.aws.dynamodb/ReplicaAutoScalingDescription))

(s/def :cognitect.aws.dynamodb/ExportStartTime inst?)

(s/def :cognitect.aws.dynamodb/TimeRangeUpperBound inst?)

(s/def :cognitect.aws.dynamodb/DescribeEndpointsRequest (s/keys))

(s/def
  :cognitect.aws.dynamodb/UpdateContinuousBackupsOutput
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.UpdateContinuousBackupsOutput/ContinuousBackupsDescription]))

(s/def
  :cognitect.aws.dynamodb/PutItemInputAttributeMap
  (s/map-of :cognitect.aws.dynamodb/AttributeName :cognitect.aws.dynamodb/AttributeValue))

(s/def
  :cognitect.aws.dynamodb/BinaryAttributeValue
  (s/or :byte-array bytes? :input-stream #(instance? java.io.InputStream %)))

(s/def
  :cognitect.aws.dynamodb/AttributeDefinition
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.AttributeDefinition/AttributeName
     :cognitect.aws.dynamodb.AttributeDefinition/AttributeType]))

(s/def
  :cognitect.aws.dynamodb/ImportArn
  (s/spec
    (s/and string? #(<= 37 (count %) 1024))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 37 1024) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.dynamodb/ConsumedCapacity
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.ConsumedCapacity/WriteCapacityUnits
     :cognitect.aws.dynamodb.ConsumedCapacity/LocalSecondaryIndexes
     :cognitect.aws.dynamodb.ConsumedCapacity/ReadCapacityUnits
     :cognitect.aws.dynamodb.ConsumedCapacity/TableName
     :cognitect.aws.dynamodb.ConsumedCapacity/GlobalSecondaryIndexes
     :cognitect.aws.dynamodb.ConsumedCapacity/Table
     :cognitect.aws.dynamodb.ConsumedCapacity/CapacityUnits]))

(s/def
  :cognitect.aws.dynamodb/TableName
  (s/spec #(re-matches (re-pattern "[a-zA-Z0-9_.-]+") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.dynamodb/KeyList
  (s/coll-of :cognitect.aws.dynamodb/Key :min-count 1 :max-count 100))

(s/def
  :cognitect.aws.dynamodb/ContributorInsightsSummaries
  (s/coll-of :cognitect.aws.dynamodb/ContributorInsightsSummary))

(s/def
  :cognitect.aws.dynamodb/AttributeName
  (s/spec
    (s/and string? #(>= 65535 (count %)))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 0 65535) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.dynamodb/ReplicaSettingsDescriptionList
  (s/coll-of :cognitect.aws.dynamodb/ReplicaSettingsDescription))

(s/def :cognitect.aws.dynamodb/NullAttributeValue boolean?)

(s/def
  :cognitect.aws.dynamodb/ScanSegment
  (s/spec (s/and int? #(<= 0 % 999999)) :gen #(gen/choose 0 999999)))

(s/def
  :cognitect.aws.dynamodb/ReplicationGroupUpdateList
  (s/coll-of :cognitect.aws.dynamodb/ReplicationGroupUpdate :min-count 1))

(s/def
  :cognitect.aws.dynamodb/KinesisStreamingDestinationInput
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.KinesisStreamingDestinationInput/TableName
     :cognitect.aws.dynamodb.KinesisStreamingDestinationInput/StreamArn]))

(s/def :cognitect.aws.dynamodb/Endpoints (s/coll-of :cognitect.aws.dynamodb/Endpoint))

(s/def :cognitect.aws.dynamodb/ExportManifest string?)

(s/def
  :cognitect.aws.dynamodb/AttributeMap
  (s/map-of :cognitect.aws.dynamodb/AttributeName :cognitect.aws.dynamodb/AttributeValue))

(s/def
  :cognitect.aws.dynamodb/GlobalSecondaryIndexUpdate
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.GlobalSecondaryIndexUpdate/Delete
     :cognitect.aws.dynamodb.GlobalSecondaryIndexUpdate/Update
     :cognitect.aws.dynamodb.GlobalSecondaryIndexUpdate/Create]))

(s/def
  :cognitect.aws.dynamodb/ListExportsOutput
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.ListExportsOutput/NextToken
     :cognitect.aws.dynamodb.ListExportsOutput/ExportSummaries]))

(s/def
  :cognitect.aws.dynamodb/KeysAndAttributes
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.KeysAndAttributes/Keys]
    :opt-un
    [:cognitect.aws.dynamodb.KeysAndAttributes/AttributesToGet
     :cognitect.aws.dynamodb.KeysAndAttributes/ConsistentRead
     :cognitect.aws.dynamodb.KeysAndAttributes/ExpressionAttributeNames
     :cognitect.aws.dynamodb.KeysAndAttributes/ProjectionExpression]))

(s/def
  :cognitect.aws.dynamodb/ItemCollectionMetrics
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.ItemCollectionMetrics/ItemCollectionKey
     :cognitect.aws.dynamodb.ItemCollectionMetrics/SizeEstimateRangeGB]))

(s/def
  :cognitect.aws.dynamodb/SSEDescription
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.SSEDescription/Status
     :cognitect.aws.dynamodb.SSEDescription/InaccessibleEncryptionDateTime
     :cognitect.aws.dynamodb.SSEDescription/KMSMasterKeyArn
     :cognitect.aws.dynamodb.SSEDescription/SSEType]))

(s/def
  :cognitect.aws.dynamodb/BackupsInputLimit
  (s/spec (s/and int? #(<= 1 % 100)) :gen #(gen/choose 1 100)))

(s/def
  :cognitect.aws.dynamodb/PartiQLBatchResponse
  (s/coll-of :cognitect.aws.dynamodb/BatchStatementResponse))

(s/def
  :cognitect.aws.dynamodb/ItemResponseList
  (s/coll-of :cognitect.aws.dynamodb/ItemResponse :min-count 1 :max-count 100))

(s/def
  :cognitect.aws.dynamodb/GlobalTableStatus
  (s/spec string? :gen #(s/gen #{"UPDATING" "DELETING" "CREATING" "ACTIVE"})))

(s/def
  :cognitect.aws.dynamodb/TransactWriteItemList
  (s/coll-of :cognitect.aws.dynamodb/TransactWriteItem :min-count 1 :max-count 100))

(s/def
  :cognitect.aws.dynamodb/NonKeyAttributeNameList
  (s/coll-of :cognitect.aws.dynamodb/NonKeyAttributeName :min-count 1 :max-count 20))

(s/def
  :cognitect.aws.dynamodb/InputFormat
  (s/spec string? :gen #(s/gen #{"CSV" "ION" "DYNAMODB_JSON"})))

(s/def
  :cognitect.aws.dynamodb/GlobalSecondaryIndex
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.GlobalSecondaryIndex/IndexName
     :cognitect.aws.dynamodb.GlobalSecondaryIndex/KeySchema
     :cognitect.aws.dynamodb.GlobalSecondaryIndex/Projection]
    :opt-un
    [:cognitect.aws.dynamodb.GlobalSecondaryIndex/ProvisionedThroughput]))

(s/def
  :cognitect.aws.dynamodb/ItemCollectionSizeEstimateRange
  (s/coll-of :cognitect.aws.dynamodb/ItemCollectionSizeEstimateBound))

(s/def
  :cognitect.aws.dynamodb/TimeToLiveAttributeName
  (s/spec
    (s/and string? #(<= 1 (count %) 255))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 1 255) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.dynamodb/ClientToken
  (s/spec #(re-matches (re-pattern "^[^\\$]+$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.dynamodb/ExportArn
  (s/spec
    (s/and string? #(<= 37 (count %) 1024))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 37 1024) #(gen/vector (gen/char-alpha) %))))))

(s/def :cognitect.aws.dynamodb/BackupSummaries (s/coll-of :cognitect.aws.dynamodb/BackupSummary))

(s/def
  :cognitect.aws.dynamodb/CreateGlobalTableInput
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.CreateGlobalTableInput/GlobalTableName
     :cognitect.aws.dynamodb.CreateGlobalTableInput/ReplicationGroup]))

(s/def
  :cognitect.aws.dynamodb/DescribeGlobalTableSettingsInput
  (s/keys :req-un [:cognitect.aws.dynamodb.DescribeGlobalTableSettingsInput/GlobalTableName]))

(s/def
  :cognitect.aws.dynamodb/GlobalSecondaryIndexInfo
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.GlobalSecondaryIndexInfo/Projection
     :cognitect.aws.dynamodb.GlobalSecondaryIndexInfo/IndexName
     :cognitect.aws.dynamodb.GlobalSecondaryIndexInfo/KeySchema
     :cognitect.aws.dynamodb.GlobalSecondaryIndexInfo/ProvisionedThroughput]))

(s/def
  :cognitect.aws.dynamodb/GlobalTable
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.GlobalTable/GlobalTableName
     :cognitect.aws.dynamodb.GlobalTable/ReplicationGroup]))

(s/def
  :cognitect.aws.dynamodb/DeleteItemOutput
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.DeleteItemOutput/ItemCollectionMetrics
     :cognitect.aws.dynamodb.DeleteItemOutput/Attributes
     :cognitect.aws.dynamodb.DeleteItemOutput/ConsumedCapacity]))

(s/def :cognitect.aws.dynamodb/Backfilling boolean?)

(s/def
  :cognitect.aws.dynamodb/LocalSecondaryIndexDescriptionList
  (s/coll-of :cognitect.aws.dynamodb/LocalSecondaryIndexDescription))

(s/def
  :cognitect.aws.dynamodb/ProjectionType
  (s/spec string? :gen #(s/gen #{"INCLUDE" "ALL" "KEYS_ONLY"})))

(s/def
  :cognitect.aws.dynamodb/Condition
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.Condition/ComparisonOperator]
    :opt-un
    [:cognitect.aws.dynamodb.Condition/AttributeValueList]))

(s/def
  :cognitect.aws.dynamodb/TimeToLiveStatus
  (s/spec string? :gen #(s/gen #{"DISABLED" "DISABLING" "ENABLING" "ENABLED"})))

(s/def
  :cognitect.aws.dynamodb/AttributeDefinitions
  (s/coll-of :cognitect.aws.dynamodb/AttributeDefinition))

(s/def
  :cognitect.aws.dynamodb/TimeToLiveDescription
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.TimeToLiveDescription/AttributeName
     :cognitect.aws.dynamodb.TimeToLiveDescription/TimeToLiveStatus]))

(s/def
  :cognitect.aws.dynamodb/ReturnValuesOnConditionCheckFailure
  (s/spec string? :gen #(s/gen #{"NONE" "ALL_OLD"})))

(s/def
  :cognitect.aws.dynamodb/KeySchemaElement
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.KeySchemaElement/AttributeName
     :cognitect.aws.dynamodb.KeySchemaElement/KeyType]))

(s/def
  :cognitect.aws.dynamodb/DeleteGlobalSecondaryIndexAction
  (s/keys :req-un [:cognitect.aws.dynamodb.DeleteGlobalSecondaryIndexAction/IndexName]))

(s/def
  :cognitect.aws.dynamodb/CreateGlobalTableOutput
  (s/keys :opt-un [:cognitect.aws.dynamodb.CreateGlobalTableOutput/GlobalTableDescription]))

(s/def
  :cognitect.aws.dynamodb/IndexStatus
  (s/spec string? :gen #(s/gen #{"UPDATING" "DELETING" "CREATING" "ACTIVE"})))

(s/def :cognitect.aws.dynamodb/DeletionProtectionEnabled boolean?)

(s/def :cognitect.aws.dynamodb/TableArn string?)

(s/def
  :cognitect.aws.dynamodb/BatchGetRequestMap
  (s/map-of
    :cognitect.aws.dynamodb/TableName
    :cognitect.aws.dynamodb/KeysAndAttributes
    :min-count
    1
    :max-count
    100))

(s/def
  :cognitect.aws.dynamodb/NumberSetAttributeValue
  (s/coll-of :cognitect.aws.dynamodb/NumberAttributeValue))

(s/def
  :cognitect.aws.dynamodb/ReplicaGlobalSecondaryIndexList
  (s/coll-of :cognitect.aws.dynamodb/ReplicaGlobalSecondaryIndex :min-count 1))

(s/def
  :cognitect.aws.dynamodb/ExportSummary
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.ExportSummary/ExportStatus
     :cognitect.aws.dynamodb.ExportSummary/ExportArn]))

(s/def
  :cognitect.aws.dynamodb/Projection
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.Projection/ProjectionType
     :cognitect.aws.dynamodb.Projection/NonKeyAttributes]))

(s/def
  :cognitect.aws.dynamodb/UpdateGlobalTableSettingsInput
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.UpdateGlobalTableSettingsInput/GlobalTableName]
    :opt-un
    [:cognitect.aws.dynamodb.UpdateGlobalTableSettingsInput/GlobalTableProvisionedWriteCapacityAutoScalingSettingsUpdate
     :cognitect.aws.dynamodb.UpdateGlobalTableSettingsInput/GlobalTableProvisionedWriteCapacityUnits
     :cognitect.aws.dynamodb.UpdateGlobalTableSettingsInput/GlobalTableGlobalSecondaryIndexSettingsUpdate
     :cognitect.aws.dynamodb.UpdateGlobalTableSettingsInput/GlobalTableBillingMode
     :cognitect.aws.dynamodb.UpdateGlobalTableSettingsInput/ReplicaSettingsUpdate]))

(s/def
  :cognitect.aws.dynamodb/DescribeContributorInsightsOutput
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.DescribeContributorInsightsOutput/ContributorInsightsRuleList
     :cognitect.aws.dynamodb.DescribeContributorInsightsOutput/IndexName
     :cognitect.aws.dynamodb.DescribeContributorInsightsOutput/ContributorInsightsStatus
     :cognitect.aws.dynamodb.DescribeContributorInsightsOutput/FailureException
     :cognitect.aws.dynamodb.DescribeContributorInsightsOutput/LastUpdateDateTime
     :cognitect.aws.dynamodb.DescribeContributorInsightsOutput/TableName]))

(s/def :cognitect.aws.dynamodb/ImportSummaryList (s/coll-of :cognitect.aws.dynamodb/ImportSummary))

(s/def
  :cognitect.aws.dynamodb/ProcessedItemCount
  (s/spec (s/and int? #(<= 0 %)) :gen #(gen/choose 0 Long/MAX_VALUE)))

(s/def
  :cognitect.aws.dynamodb/TableCreationParameters
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.TableCreationParameters/TableName
     :cognitect.aws.dynamodb.TableCreationParameters/AttributeDefinitions
     :cognitect.aws.dynamodb.TableCreationParameters/KeySchema]
    :opt-un
    [:cognitect.aws.dynamodb.TableCreationParameters/SSESpecification
     :cognitect.aws.dynamodb.TableCreationParameters/BillingMode
     :cognitect.aws.dynamodb.TableCreationParameters/GlobalSecondaryIndexes
     :cognitect.aws.dynamodb.TableCreationParameters/ProvisionedThroughput]))

(s/def
  :cognitect.aws.dynamodb/ScanOutput
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.ScanOutput/Count
     :cognitect.aws.dynamodb.ScanOutput/LastEvaluatedKey
     :cognitect.aws.dynamodb.ScanOutput/ScannedCount
     :cognitect.aws.dynamodb.ScanOutput/ConsumedCapacity
     :cognitect.aws.dynamodb.ScanOutput/Items]))

(s/def
  :cognitect.aws.dynamodb/BackupArn
  (s/spec
    (s/and string? #(<= 37 (count %) 1024))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 37 1024) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.dynamodb/Select
  (s/spec
    string?
    :gen
    #(s/gen #{"COUNT" "ALL_PROJECTED_ATTRIBUTES" "ALL_ATTRIBUTES" "SPECIFIC_ATTRIBUTES"})))

(s/def
  :cognitect.aws.dynamodb/ReplicaGlobalSecondaryIndexAutoScalingUpdate
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndexAutoScalingUpdate/IndexName
     :cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndexAutoScalingUpdate/ProvisionedReadCapacityAutoScalingUpdate]))

(s/def :cognitect.aws.dynamodb/ReplicaUpdateList (s/coll-of :cognitect.aws.dynamodb/ReplicaUpdate))

(s/def
  :cognitect.aws.dynamodb/ReplicaAutoScalingDescription
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.ReplicaAutoScalingDescription/ReplicaStatus
     :cognitect.aws.dynamodb.ReplicaAutoScalingDescription/ReplicaProvisionedReadCapacityAutoScalingSettings
     :cognitect.aws.dynamodb.ReplicaAutoScalingDescription/RegionName
     :cognitect.aws.dynamodb.ReplicaAutoScalingDescription/GlobalSecondaryIndexes
     :cognitect.aws.dynamodb.ReplicaAutoScalingDescription/ReplicaProvisionedWriteCapacityAutoScalingSettings]))

(s/def :cognitect.aws.dynamodb/KeyType (s/spec string? :gen #(s/gen #{"RANGE" "HASH"})))

(s/def
  :cognitect.aws.dynamodb/ContinuousBackupsDescription
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.ContinuousBackupsDescription/ContinuousBackupsStatus]
    :opt-un
    [:cognitect.aws.dynamodb.ContinuousBackupsDescription/PointInTimeRecoveryDescription]))

(s/def
  :cognitect.aws.dynamodb/ListImportsMaxLimit
  (s/spec (s/and int? #(<= 1 % 25)) :gen #(gen/choose 1 25)))

(s/def
  :cognitect.aws.dynamodb/ReturnValue
  (s/spec string? :gen #(s/gen #{"UPDATED_OLD" "ALL_NEW" "NONE" "UPDATED_NEW" "ALL_OLD"})))

(s/def
  :cognitect.aws.dynamodb/UpdateContinuousBackupsInput
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.UpdateContinuousBackupsInput/TableName
     :cognitect.aws.dynamodb.UpdateContinuousBackupsInput/PointInTimeRecoverySpecification]))

(s/def
  :cognitect.aws.dynamodb/Capacity
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.Capacity/WriteCapacityUnits
     :cognitect.aws.dynamodb.Capacity/ReadCapacityUnits
     :cognitect.aws.dynamodb.Capacity/CapacityUnits]))

(s/def
  :cognitect.aws.dynamodb/GetItemOutput
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.GetItemOutput/ConsumedCapacity
     :cognitect.aws.dynamodb.GetItemOutput/Item]))

(s/def
  :cognitect.aws.dynamodb/IndexName
  (s/spec #(re-matches (re-pattern "[a-zA-Z0-9_.-]+") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.dynamodb/DescribeExportOutput
  (s/keys :opt-un [:cognitect.aws.dynamodb.DescribeExportOutput/ExportDescription]))

(s/def
  :cognitect.aws.dynamodb/ReplicaGlobalSecondaryIndexAutoScalingDescription
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndexAutoScalingDescription/IndexStatus
     :cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndexAutoScalingDescription/IndexName
     :cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndexAutoScalingDescription/ProvisionedWriteCapacityAutoScalingSettings
     :cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndexAutoScalingDescription/ProvisionedReadCapacityAutoScalingSettings]))

(s/def
  :cognitect.aws.dynamodb/ListGlobalTablesInput
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.ListGlobalTablesInput/ExclusiveStartGlobalTableName
     :cognitect.aws.dynamodb.ListGlobalTablesInput/Limit
     :cognitect.aws.dynamodb.ListGlobalTablesInput/RegionName]))

(s/def
  :cognitect.aws.dynamodb/FilterConditionMap
  (s/map-of :cognitect.aws.dynamodb/AttributeName :cognitect.aws.dynamodb/Condition))

(s/def
  :cognitect.aws.dynamodb/ExpectedAttributeMap
  (s/map-of :cognitect.aws.dynamodb/AttributeName :cognitect.aws.dynamodb/ExpectedAttributeValue))

(s/def
  :cognitect.aws.dynamodb/DescribeTimeToLiveInput
  (s/keys :req-un [:cognitect.aws.dynamodb.DescribeTimeToLiveInput/TableName]))

(s/def
  :cognitect.aws.dynamodb/GlobalSecondaryIndexAutoScalingUpdateList
  (s/coll-of :cognitect.aws.dynamodb/GlobalSecondaryIndexAutoScalingUpdate :min-count 1))

(s/def
  :cognitect.aws.dynamodb/UpdateContributorInsightsInput
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.UpdateContributorInsightsInput/TableName
     :cognitect.aws.dynamodb.UpdateContributorInsightsInput/ContributorInsightsAction]
    :opt-un
    [:cognitect.aws.dynamodb.UpdateContributorInsightsInput/IndexName]))

(s/def
  :cognitect.aws.dynamodb/ListBackupsInput
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.ListBackupsInput/TimeRangeUpperBound
     :cognitect.aws.dynamodb.ListBackupsInput/Limit
     :cognitect.aws.dynamodb.ListBackupsInput/ExclusiveStartBackupArn
     :cognitect.aws.dynamodb.ListBackupsInput/TimeRangeLowerBound
     :cognitect.aws.dynamodb.ListBackupsInput/TableName
     :cognitect.aws.dynamodb.ListBackupsInput/BackupType]))

(s/def
  :cognitect.aws.dynamodb/AttributeValue
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.AttributeValue/NULL
     :cognitect.aws.dynamodb.AttributeValue/L
     :cognitect.aws.dynamodb.AttributeValue/BOOL
     :cognitect.aws.dynamodb.AttributeValue/M
     :cognitect.aws.dynamodb.AttributeValue/S
     :cognitect.aws.dynamodb.AttributeValue/B
     :cognitect.aws.dynamodb.AttributeValue/NS
     :cognitect.aws.dynamodb.AttributeValue/N
     :cognitect.aws.dynamodb.AttributeValue/BS
     :cognitect.aws.dynamodb.AttributeValue/SS]))

(s/def
  :cognitect.aws.dynamodb/ItemCollectionMetricsPerTable
  (s/map-of
    :cognitect.aws.dynamodb/TableName
    :cognitect.aws.dynamodb/ItemCollectionMetricsMultiple))

(s/def :cognitect.aws.dynamodb/String string?)

(s/def :cognitect.aws.dynamodb/ItemList (s/coll-of :cognitect.aws.dynamodb/AttributeMap))

(s/def :cognitect.aws.dynamodb/ConditionalOperator (s/spec string? :gen #(s/gen #{"AND" "OR"})))

(s/def
  :cognitect.aws.dynamodb/ListTagsOfResourceInput
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.ListTagsOfResourceInput/ResourceArn]
    :opt-un
    [:cognitect.aws.dynamodb.ListTagsOfResourceInput/NextToken]))

(s/def
  :cognitect.aws.dynamodb/PointInTimeRecoveryStatus
  (s/spec string? :gen #(s/gen #{"DISABLED" "ENABLED"})))

(s/def
  :cognitect.aws.dynamodb/KeyConditions
  (s/map-of :cognitect.aws.dynamodb/AttributeName :cognitect.aws.dynamodb/Condition))

(s/def
  :cognitect.aws.dynamodb/UpdateItemInput
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.UpdateItemInput/TableName :cognitect.aws.dynamodb.UpdateItemInput/Key]
    :opt-un
    [:cognitect.aws.dynamodb.UpdateItemInput/Expected
     :cognitect.aws.dynamodb.UpdateItemInput/AttributeUpdates
     :cognitect.aws.dynamodb.UpdateItemInput/ReturnValuesOnConditionCheckFailure
     :cognitect.aws.dynamodb.UpdateItemInput/ReturnConsumedCapacity
     :cognitect.aws.dynamodb.UpdateItemInput/ConditionalOperator
     :cognitect.aws.dynamodb.UpdateItemInput/UpdateExpression
     :cognitect.aws.dynamodb.UpdateItemInput/ConditionExpression
     :cognitect.aws.dynamodb.UpdateItemInput/ExpressionAttributeNames
     :cognitect.aws.dynamodb.UpdateItemInput/ReturnValues
     :cognitect.aws.dynamodb.UpdateItemInput/ExpressionAttributeValues
     :cognitect.aws.dynamodb.UpdateItemInput/ReturnItemCollectionMetrics]))

(s/def
  :cognitect.aws.dynamodb/GetItemInput
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.GetItemInput/TableName :cognitect.aws.dynamodb.GetItemInput/Key]
    :opt-un
    [:cognitect.aws.dynamodb.GetItemInput/AttributesToGet
     :cognitect.aws.dynamodb.GetItemInput/ConsistentRead
     :cognitect.aws.dynamodb.GetItemInput/ReturnConsumedCapacity
     :cognitect.aws.dynamodb.GetItemInput/ExpressionAttributeNames
     :cognitect.aws.dynamodb.GetItemInput/ProjectionExpression]))

(s/def
  :cognitect.aws.dynamodb/Replica
  (s/keys :opt-un [:cognitect.aws.dynamodb.Replica/RegionName]))

(s/def
  :cognitect.aws.dynamodb/BackupTypeFilter
  (s/spec string? :gen #(s/gen #{"SYSTEM" "USER" "AWS_BACKUP" "ALL"})))

(s/def :cognitect.aws.dynamodb/TagKeyList (s/coll-of :cognitect.aws.dynamodb/TagKeyString))

(s/def
  :cognitect.aws.dynamodb/DescribeTableReplicaAutoScalingOutput
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.DescribeTableReplicaAutoScalingOutput/TableAutoScalingDescription]))

(s/def
  :cognitect.aws.dynamodb/DescribeExportInput
  (s/keys :req-un [:cognitect.aws.dynamodb.DescribeExportInput/ExportArn]))

(s/def
  :cognitect.aws.dynamodb/InputCompressionType
  (s/spec string? :gen #(s/gen #{"GZIP" "NONE" "ZSTD"})))

(s/def
  :cognitect.aws.dynamodb/RestoreTableFromBackupInput
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.RestoreTableFromBackupInput/TargetTableName
     :cognitect.aws.dynamodb.RestoreTableFromBackupInput/BackupArn]
    :opt-un
    [:cognitect.aws.dynamodb.RestoreTableFromBackupInput/BillingModeOverride
     :cognitect.aws.dynamodb.RestoreTableFromBackupInput/ProvisionedThroughputOverride
     :cognitect.aws.dynamodb.RestoreTableFromBackupInput/LocalSecondaryIndexOverride
     :cognitect.aws.dynamodb.RestoreTableFromBackupInput/GlobalSecondaryIndexOverride
     :cognitect.aws.dynamodb.RestoreTableFromBackupInput/SSESpecificationOverride]))

(s/def
  :cognitect.aws.dynamodb/DeleteReplicaAction
  (s/keys :req-un [:cognitect.aws.dynamodb.DeleteReplicaAction/RegionName]))

(s/def
  :cognitect.aws.dynamodb/BillingModeSummary
  (s/keys
    :opt-un
    [:cognitect.aws.dynamodb.BillingModeSummary/LastUpdateToPayPerRequestDateTime
     :cognitect.aws.dynamodb.BillingModeSummary/BillingMode]))

(s/def
  :cognitect.aws.dynamodb/TimeToLiveSpecification
  (s/keys
    :req-un
    [:cognitect.aws.dynamodb.TimeToLiveSpecification/Enabled
     :cognitect.aws.dynamodb.TimeToLiveSpecification/AttributeName]))

(s/def
  :cognitect.aws.dynamodb/ItemResponse
  (s/keys :opt-un [:cognitect.aws.dynamodb.ItemResponse/Item]))

(s/def :cognitect.aws.dynamodb.TransactGetItem/Get :cognitect.aws.dynamodb/Get)

(s/def
  :cognitect.aws.dynamodb.BatchStatementResponse/Error
  :cognitect.aws.dynamodb/BatchStatementError)

(s/def :cognitect.aws.dynamodb.BatchStatementResponse/TableName :cognitect.aws.dynamodb/TableName)

(s/def :cognitect.aws.dynamodb.BatchStatementResponse/Item :cognitect.aws.dynamodb/AttributeMap)

(s/def
  :cognitect.aws.dynamodb.AutoScalingSettingsDescription/MinimumUnits
  :cognitect.aws.dynamodb/PositiveLongObject)

(s/def
  :cognitect.aws.dynamodb.AutoScalingSettingsDescription/MaximumUnits
  :cognitect.aws.dynamodb/PositiveLongObject)

(s/def
  :cognitect.aws.dynamodb.AutoScalingSettingsDescription/AutoScalingDisabled
  :cognitect.aws.dynamodb/BooleanObject)

(s/def
  :cognitect.aws.dynamodb.AutoScalingSettingsDescription/AutoScalingRoleArn
  :cognitect.aws.dynamodb/String)

(s/def
  :cognitect.aws.dynamodb.AutoScalingSettingsDescription/ScalingPolicies
  :cognitect.aws.dynamodb/AutoScalingPolicyDescriptionList)

(s/def :cognitect.aws.dynamodb.Update/Key :cognitect.aws.dynamodb/Key)

(s/def :cognitect.aws.dynamodb.Update/UpdateExpression :cognitect.aws.dynamodb/UpdateExpression)

(s/def :cognitect.aws.dynamodb.Update/TableName :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.Update/ConditionExpression
  :cognitect.aws.dynamodb/ConditionExpression)

(s/def
  :cognitect.aws.dynamodb.Update/ExpressionAttributeNames
  :cognitect.aws.dynamodb/ExpressionAttributeNameMap)

(s/def
  :cognitect.aws.dynamodb.Update/ExpressionAttributeValues
  :cognitect.aws.dynamodb/ExpressionAttributeValueMap)

(s/def
  :cognitect.aws.dynamodb.Update/ReturnValuesOnConditionCheckFailure
  :cognitect.aws.dynamodb/ReturnValuesOnConditionCheckFailure)

(s/def :cognitect.aws.dynamodb.DeleteBackupInput/BackupArn :cognitect.aws.dynamodb/BackupArn)

(s/def :cognitect.aws.dynamodb.ImportSummary/ImportArn :cognitect.aws.dynamodb/ImportArn)

(s/def :cognitect.aws.dynamodb.ImportSummary/ImportStatus :cognitect.aws.dynamodb/ImportStatus)

(s/def :cognitect.aws.dynamodb.ImportSummary/TableArn :cognitect.aws.dynamodb/TableArn)

(s/def :cognitect.aws.dynamodb.ImportSummary/S3BucketSource :cognitect.aws.dynamodb/S3BucketSource)

(s/def
  :cognitect.aws.dynamodb.ImportSummary/CloudWatchLogGroupArn
  :cognitect.aws.dynamodb/CloudWatchLogGroupArn)

(s/def :cognitect.aws.dynamodb.ImportSummary/InputFormat :cognitect.aws.dynamodb/InputFormat)

(s/def :cognitect.aws.dynamodb.ImportSummary/StartTime :cognitect.aws.dynamodb/ImportStartTime)

(s/def :cognitect.aws.dynamodb.ImportSummary/EndTime :cognitect.aws.dynamodb/ImportEndTime)

(s/def
  :cognitect.aws.dynamodb.ListImportsOutput/ImportSummaryList
  :cognitect.aws.dynamodb/ImportSummaryList)

(s/def :cognitect.aws.dynamodb.ListImportsOutput/NextToken :cognitect.aws.dynamodb/ImportNextToken)

(s/def
  :cognitect.aws.dynamodb.DescribeKinesisStreamingDestinationOutput/TableName
  :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.DescribeKinesisStreamingDestinationOutput/KinesisDataStreamDestinations
  :cognitect.aws.dynamodb/KinesisDataStreamDestinations)

(s/def
  :cognitect.aws.dynamodb.ListContributorInsightsInput/TableName
  :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.ListContributorInsightsInput/NextToken
  :cognitect.aws.dynamodb/NextTokenString)

(s/def
  :cognitect.aws.dynamodb.ListContributorInsightsInput/MaxResults
  :cognitect.aws.dynamodb/ListContributorInsightsLimit)

(s/def
  :cognitect.aws.dynamodb.DescribeContinuousBackupsInput/TableName
  :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.ListBackupsOutput/BackupSummaries
  :cognitect.aws.dynamodb/BackupSummaries)

(s/def
  :cognitect.aws.dynamodb.ListBackupsOutput/LastEvaluatedBackupArn
  :cognitect.aws.dynamodb/BackupArn)

(s/def
  :cognitect.aws.dynamodb.DescribeTableReplicaAutoScalingInput/TableName
  :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.DeleteReplicationGroupMemberAction/RegionName
  :cognitect.aws.dynamodb/RegionName)

(s/def :cognitect.aws.dynamodb.S3BucketSource/S3BucketOwner :cognitect.aws.dynamodb/S3BucketOwner)

(s/def :cognitect.aws.dynamodb.S3BucketSource/S3Bucket :cognitect.aws.dynamodb/S3Bucket)

(s/def :cognitect.aws.dynamodb.S3BucketSource/S3KeyPrefix :cognitect.aws.dynamodb/S3Prefix)

(s/def
  :cognitect.aws.dynamodb.DescribeGlobalTableOutput/GlobalTableDescription
  :cognitect.aws.dynamodb/GlobalTableDescription)

(s/def :cognitect.aws.dynamodb.UpdateItemOutput/Attributes :cognitect.aws.dynamodb/AttributeMap)

(s/def
  :cognitect.aws.dynamodb.UpdateItemOutput/ConsumedCapacity
  :cognitect.aws.dynamodb/ConsumedCapacity)

(s/def
  :cognitect.aws.dynamodb.UpdateItemOutput/ItemCollectionMetrics
  :cognitect.aws.dynamodb/ItemCollectionMetrics)

(s/def :cognitect.aws.dynamodb.ListExportsInput/TableArn :cognitect.aws.dynamodb/TableArn)

(s/def
  :cognitect.aws.dynamodb.ListExportsInput/MaxResults
  :cognitect.aws.dynamodb/ListExportsMaxLimit)

(s/def :cognitect.aws.dynamodb.ListExportsInput/NextToken :cognitect.aws.dynamodb/ExportNextToken)

(s/def :cognitect.aws.dynamodb.LocalSecondaryIndex/IndexName :cognitect.aws.dynamodb/IndexName)

(s/def :cognitect.aws.dynamodb.LocalSecondaryIndex/KeySchema :cognitect.aws.dynamodb/KeySchema)

(s/def :cognitect.aws.dynamodb.LocalSecondaryIndex/Projection :cognitect.aws.dynamodb/Projection)

(s/def
  :cognitect.aws.dynamodb.ListGlobalTablesOutput/GlobalTables
  :cognitect.aws.dynamodb/GlobalTableList)

(s/def
  :cognitect.aws.dynamodb.ListGlobalTablesOutput/LastEvaluatedGlobalTableName
  :cognitect.aws.dynamodb/TableName)

(s/def :cognitect.aws.dynamodb.ReplicaDescription/RegionName :cognitect.aws.dynamodb/RegionName)

(s/def
  :cognitect.aws.dynamodb.ReplicaDescription/ReplicaStatusDescription
  :cognitect.aws.dynamodb/ReplicaStatusDescription)

(s/def
  :cognitect.aws.dynamodb.ReplicaDescription/ReplicaStatusPercentProgress
  :cognitect.aws.dynamodb/ReplicaStatusPercentProgress)

(s/def
  :cognitect.aws.dynamodb.ReplicaDescription/ReplicaTableClassSummary
  :cognitect.aws.dynamodb/TableClassSummary)

(s/def
  :cognitect.aws.dynamodb.ReplicaDescription/ProvisionedThroughputOverride
  :cognitect.aws.dynamodb/ProvisionedThroughputOverride)

(s/def
  :cognitect.aws.dynamodb.ReplicaDescription/ReplicaStatus
  :cognitect.aws.dynamodb/ReplicaStatus)

(s/def
  :cognitect.aws.dynamodb.ReplicaDescription/GlobalSecondaryIndexes
  :cognitect.aws.dynamodb/ReplicaGlobalSecondaryIndexDescriptionList)

(s/def
  :cognitect.aws.dynamodb.ReplicaDescription/KMSMasterKeyId
  :cognitect.aws.dynamodb/KMSMasterKeyId)

(s/def
  :cognitect.aws.dynamodb.ReplicaDescription/ReplicaInaccessibleDateTime
  :cognitect.aws.dynamodb/Date)

(s/def
  :cognitect.aws.dynamodb.CreateBackupOutput/BackupDetails
  :cognitect.aws.dynamodb/BackupDetails)

(s/def :cognitect.aws.dynamodb.Tag/Key :cognitect.aws.dynamodb/TagKeyString)

(s/def :cognitect.aws.dynamodb.Tag/Value :cognitect.aws.dynamodb/TagValueString)

(s/def :cognitect.aws.dynamodb.ListTagsOfResourceOutput/Tags :cognitect.aws.dynamodb/TagList)

(s/def
  :cognitect.aws.dynamodb.ListTagsOfResourceOutput/NextToken
  :cognitect.aws.dynamodb/NextTokenString)

(s/def
  :cognitect.aws.dynamodb.BatchExecuteStatementOutput/Responses
  :cognitect.aws.dynamodb/PartiQLBatchResponse)

(s/def
  :cognitect.aws.dynamodb.BatchExecuteStatementOutput/ConsumedCapacity
  :cognitect.aws.dynamodb/ConsumedCapacityMultiple)

(s/def :cognitect.aws.dynamodb.ReplicaSettingsUpdate/RegionName :cognitect.aws.dynamodb/RegionName)

(s/def
  :cognitect.aws.dynamodb.ReplicaSettingsUpdate/ReplicaProvisionedReadCapacityUnits
  :cognitect.aws.dynamodb/PositiveLongObject)

(s/def
  :cognitect.aws.dynamodb.ReplicaSettingsUpdate/ReplicaProvisionedReadCapacityAutoScalingSettingsUpdate
  :cognitect.aws.dynamodb/AutoScalingSettingsUpdate)

(s/def
  :cognitect.aws.dynamodb.ReplicaSettingsUpdate/ReplicaGlobalSecondaryIndexSettingsUpdate
  :cognitect.aws.dynamodb/ReplicaGlobalSecondaryIndexSettingsUpdateList)

(s/def
  :cognitect.aws.dynamodb.ReplicaSettingsUpdate/ReplicaTableClass
  :cognitect.aws.dynamodb/TableClass)

(s/def :cognitect.aws.dynamodb.SSESpecification/Enabled :cognitect.aws.dynamodb/SSEEnabled)

(s/def :cognitect.aws.dynamodb.SSESpecification/SSEType :cognitect.aws.dynamodb/SSEType)

(s/def
  :cognitect.aws.dynamodb.SSESpecification/KMSMasterKeyId
  :cognitect.aws.dynamodb/KMSMasterKeyId)

(s/def
  :cognitect.aws.dynamodb.TableAutoScalingDescription/TableName
  :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.TableAutoScalingDescription/TableStatus
  :cognitect.aws.dynamodb/TableStatus)

(s/def
  :cognitect.aws.dynamodb.TableAutoScalingDescription/Replicas
  :cognitect.aws.dynamodb/ReplicaAutoScalingDescriptionList)

(s/def
  :cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndexSettingsUpdate/IndexName
  :cognitect.aws.dynamodb/IndexName)

(s/def
  :cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndexSettingsUpdate/ProvisionedReadCapacityUnits
  :cognitect.aws.dynamodb/PositiveLongObject)

(s/def
  :cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndexSettingsUpdate/ProvisionedReadCapacityAutoScalingSettingsUpdate
  :cognitect.aws.dynamodb/AutoScalingSettingsUpdate)

(s/def
  :cognitect.aws.dynamodb.UpdateTableReplicaAutoScalingInput/GlobalSecondaryIndexUpdates
  :cognitect.aws.dynamodb/GlobalSecondaryIndexAutoScalingUpdateList)

(s/def
  :cognitect.aws.dynamodb.UpdateTableReplicaAutoScalingInput/TableName
  :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.UpdateTableReplicaAutoScalingInput/ProvisionedWriteCapacityAutoScalingUpdate
  :cognitect.aws.dynamodb/AutoScalingSettingsUpdate)

(s/def
  :cognitect.aws.dynamodb.UpdateTableReplicaAutoScalingInput/ReplicaUpdates
  :cognitect.aws.dynamodb/ReplicaAutoScalingUpdateList)

(s/def
  :cognitect.aws.dynamodb.ReplicaAutoScalingUpdate/RegionName
  :cognitect.aws.dynamodb/RegionName)

(s/def
  :cognitect.aws.dynamodb.ReplicaAutoScalingUpdate/ReplicaGlobalSecondaryIndexUpdates
  :cognitect.aws.dynamodb/ReplicaGlobalSecondaryIndexAutoScalingUpdateList)

(s/def
  :cognitect.aws.dynamodb.ReplicaAutoScalingUpdate/ReplicaProvisionedReadCapacityAutoScalingUpdate
  :cognitect.aws.dynamodb/AutoScalingSettingsUpdate)

(s/def :cognitect.aws.dynamodb.ReplicaUpdate/Create :cognitect.aws.dynamodb/CreateReplicaAction)

(s/def :cognitect.aws.dynamodb.ReplicaUpdate/Delete :cognitect.aws.dynamodb/DeleteReplicaAction)

(s/def
  :cognitect.aws.dynamodb.DescribeImportOutput/ImportTableDescription
  :cognitect.aws.dynamodb/ImportTableDescription)

(s/def
  :cognitect.aws.dynamodb.UpdateContributorInsightsOutput/TableName
  :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.UpdateContributorInsightsOutput/IndexName
  :cognitect.aws.dynamodb/IndexName)

(s/def
  :cognitect.aws.dynamodb.UpdateContributorInsightsOutput/ContributorInsightsStatus
  :cognitect.aws.dynamodb/ContributorInsightsStatus)

(s/def
  :cognitect.aws.dynamodb.DescribeContinuousBackupsOutput/ContinuousBackupsDescription
  :cognitect.aws.dynamodb/ContinuousBackupsDescription)

(s/def
  :cognitect.aws.dynamodb.DescribeBackupOutput/BackupDescription
  :cognitect.aws.dynamodb/BackupDescription)

(s/def :cognitect.aws.dynamodb.CsvOptions/Delimiter :cognitect.aws.dynamodb/CsvDelimiter)

(s/def :cognitect.aws.dynamodb.CsvOptions/HeaderList :cognitect.aws.dynamodb/CsvHeaderList)

(s/def
  :cognitect.aws.dynamodb.AutoScalingSettingsUpdate/MinimumUnits
  :cognitect.aws.dynamodb/PositiveLongObject)

(s/def
  :cognitect.aws.dynamodb.AutoScalingSettingsUpdate/MaximumUnits
  :cognitect.aws.dynamodb/PositiveLongObject)

(s/def
  :cognitect.aws.dynamodb.AutoScalingSettingsUpdate/AutoScalingDisabled
  :cognitect.aws.dynamodb/BooleanObject)

(s/def
  :cognitect.aws.dynamodb.AutoScalingSettingsUpdate/AutoScalingRoleArn
  :cognitect.aws.dynamodb/AutoScalingRoleArn)

(s/def
  :cognitect.aws.dynamodb.AutoScalingSettingsUpdate/ScalingPolicyUpdate
  :cognitect.aws.dynamodb/AutoScalingPolicyUpdate)

(s/def
  :cognitect.aws.dynamodb.ImportTableDescription/CloudWatchLogGroupArn
  :cognitect.aws.dynamodb/CloudWatchLogGroupArn)

(s/def :cognitect.aws.dynamodb.ImportTableDescription/EndTime :cognitect.aws.dynamodb/ImportEndTime)

(s/def
  :cognitect.aws.dynamodb.ImportTableDescription/S3BucketSource
  :cognitect.aws.dynamodb/S3BucketSource)

(s/def
  :cognitect.aws.dynamodb.ImportTableDescription/ImportedItemCount
  :cognitect.aws.dynamodb/ImportedItemCount)

(s/def
  :cognitect.aws.dynamodb.ImportTableDescription/FailureMessage
  :cognitect.aws.dynamodb/FailureMessage)

(s/def
  :cognitect.aws.dynamodb.ImportTableDescription/ImportStatus
  :cognitect.aws.dynamodb/ImportStatus)

(s/def :cognitect.aws.dynamodb.ImportTableDescription/TableId :cognitect.aws.dynamodb/TableId)

(s/def
  :cognitect.aws.dynamodb.ImportTableDescription/InputFormatOptions
  :cognitect.aws.dynamodb/InputFormatOptions)

(s/def
  :cognitect.aws.dynamodb.ImportTableDescription/StartTime
  :cognitect.aws.dynamodb/ImportStartTime)

(s/def
  :cognitect.aws.dynamodb.ImportTableDescription/FailureCode
  :cognitect.aws.dynamodb/FailureCode)

(s/def
  :cognitect.aws.dynamodb.ImportTableDescription/ProcessedSizeBytes
  :cognitect.aws.dynamodb/LongObject)

(s/def :cognitect.aws.dynamodb.ImportTableDescription/ErrorCount :cognitect.aws.dynamodb/ErrorCount)

(s/def :cognitect.aws.dynamodb.ImportTableDescription/ImportArn :cognitect.aws.dynamodb/ImportArn)

(s/def
  :cognitect.aws.dynamodb.ImportTableDescription/InputFormat
  :cognitect.aws.dynamodb/InputFormat)

(s/def
  :cognitect.aws.dynamodb.ImportTableDescription/ClientToken
  :cognitect.aws.dynamodb/ClientToken)

(s/def :cognitect.aws.dynamodb.ImportTableDescription/TableArn :cognitect.aws.dynamodb/TableArn)

(s/def
  :cognitect.aws.dynamodb.ImportTableDescription/ProcessedItemCount
  :cognitect.aws.dynamodb/ProcessedItemCount)

(s/def
  :cognitect.aws.dynamodb.ImportTableDescription/TableCreationParameters
  :cognitect.aws.dynamodb/TableCreationParameters)

(s/def
  :cognitect.aws.dynamodb.ImportTableDescription/InputCompressionType
  :cognitect.aws.dynamodb/InputCompressionType)

(s/def
  :cognitect.aws.dynamodb.ParameterizedStatement/Statement
  :cognitect.aws.dynamodb/PartiQLStatement)

(s/def
  :cognitect.aws.dynamodb.ParameterizedStatement/Parameters
  :cognitect.aws.dynamodb/PreparedStatementParameters)

(s/def
  :cognitect.aws.dynamodb.ParameterizedStatement/ReturnValuesOnConditionCheckFailure
  :cognitect.aws.dynamodb/ReturnValuesOnConditionCheckFailure)

(s/def
  :cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndexSettingsDescription/IndexName
  :cognitect.aws.dynamodb/IndexName)

(s/def
  :cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndexSettingsDescription/IndexStatus
  :cognitect.aws.dynamodb/IndexStatus)

(s/def
  :cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndexSettingsDescription/ProvisionedReadCapacityUnits
  :cognitect.aws.dynamodb/PositiveLongObject)

(s/def
  :cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndexSettingsDescription/ProvisionedReadCapacityAutoScalingSettings
  :cognitect.aws.dynamodb/AutoScalingSettingsDescription)

(s/def
  :cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndexSettingsDescription/ProvisionedWriteCapacityUnits
  :cognitect.aws.dynamodb/PositiveLongObject)

(s/def
  :cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndexSettingsDescription/ProvisionedWriteCapacityAutoScalingSettings
  :cognitect.aws.dynamodb/AutoScalingSettingsDescription)

(s/def
  :cognitect.aws.dynamodb.CreateTableOutput/TableDescription
  :cognitect.aws.dynamodb/TableDescription)

(s/def :cognitect.aws.dynamodb.ListTablesOutput/TableNames :cognitect.aws.dynamodb/TableNameList)

(s/def
  :cognitect.aws.dynamodb.ListTablesOutput/LastEvaluatedTableName
  :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.BatchWriteItemOutput/UnprocessedItems
  :cognitect.aws.dynamodb/BatchWriteItemRequestMap)

(s/def
  :cognitect.aws.dynamodb.BatchWriteItemOutput/ItemCollectionMetrics
  :cognitect.aws.dynamodb/ItemCollectionMetricsPerTable)

(s/def
  :cognitect.aws.dynamodb.BatchWriteItemOutput/ConsumedCapacity
  :cognitect.aws.dynamodb/ConsumedCapacityMultiple)

(s/def :cognitect.aws.dynamodb.PutRequest/Item :cognitect.aws.dynamodb/PutItemInputAttributeMap)

(s/def
  :cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndex/IndexName
  :cognitect.aws.dynamodb/IndexName)

(s/def
  :cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndex/ProvisionedThroughputOverride
  :cognitect.aws.dynamodb/ProvisionedThroughputOverride)

(s/def :cognitect.aws.dynamodb.QueryOutput/Items :cognitect.aws.dynamodb/ItemList)

(s/def :cognitect.aws.dynamodb.QueryOutput/Count :cognitect.aws.dynamodb/Integer)

(s/def :cognitect.aws.dynamodb.QueryOutput/ScannedCount :cognitect.aws.dynamodb/Integer)

(s/def :cognitect.aws.dynamodb.QueryOutput/LastEvaluatedKey :cognitect.aws.dynamodb/Key)

(s/def
  :cognitect.aws.dynamodb.QueryOutput/ConsumedCapacity
  :cognitect.aws.dynamodb/ConsumedCapacity)

(s/def :cognitect.aws.dynamodb.Endpoint/Address :cognitect.aws.dynamodb/String)

(s/def :cognitect.aws.dynamodb.Endpoint/CachePeriodInMinutes :cognitect.aws.dynamodb/Long)

(s/def
  :cognitect.aws.dynamodb.ListContributorInsightsOutput/ContributorInsightsSummaries
  :cognitect.aws.dynamodb/ContributorInsightsSummaries)

(s/def
  :cognitect.aws.dynamodb.ListContributorInsightsOutput/NextToken
  :cognitect.aws.dynamodb/NextTokenString)

(s/def
  :cognitect.aws.dynamodb.ReplicaSettingsDescription/RegionName
  :cognitect.aws.dynamodb/RegionName)

(s/def
  :cognitect.aws.dynamodb.ReplicaSettingsDescription/ReplicaTableClassSummary
  :cognitect.aws.dynamodb/TableClassSummary)

(s/def
  :cognitect.aws.dynamodb.ReplicaSettingsDescription/ReplicaStatus
  :cognitect.aws.dynamodb/ReplicaStatus)

(s/def
  :cognitect.aws.dynamodb.ReplicaSettingsDescription/ReplicaGlobalSecondaryIndexSettings
  :cognitect.aws.dynamodb/ReplicaGlobalSecondaryIndexSettingsDescriptionList)

(s/def
  :cognitect.aws.dynamodb.ReplicaSettingsDescription/ReplicaProvisionedWriteCapacityAutoScalingSettings
  :cognitect.aws.dynamodb/AutoScalingSettingsDescription)

(s/def
  :cognitect.aws.dynamodb.ReplicaSettingsDescription/ReplicaProvisionedReadCapacityAutoScalingSettings
  :cognitect.aws.dynamodb/AutoScalingSettingsDescription)

(s/def
  :cognitect.aws.dynamodb.ReplicaSettingsDescription/ReplicaProvisionedWriteCapacityUnits
  :cognitect.aws.dynamodb/NonNegativeLongObject)

(s/def
  :cognitect.aws.dynamodb.ReplicaSettingsDescription/ReplicaBillingModeSummary
  :cognitect.aws.dynamodb/BillingModeSummary)

(s/def
  :cognitect.aws.dynamodb.ReplicaSettingsDescription/ReplicaProvisionedReadCapacityUnits
  :cognitect.aws.dynamodb/NonNegativeLongObject)

(s/def
  :cognitect.aws.dynamodb.GlobalTableDescription/ReplicationGroup
  :cognitect.aws.dynamodb/ReplicaDescriptionList)

(s/def
  :cognitect.aws.dynamodb.GlobalTableDescription/GlobalTableArn
  :cognitect.aws.dynamodb/GlobalTableArnString)

(s/def :cognitect.aws.dynamodb.GlobalTableDescription/CreationDateTime :cognitect.aws.dynamodb/Date)

(s/def
  :cognitect.aws.dynamodb.GlobalTableDescription/GlobalTableStatus
  :cognitect.aws.dynamodb/GlobalTableStatus)

(s/def
  :cognitect.aws.dynamodb.GlobalTableDescription/GlobalTableName
  :cognitect.aws.dynamodb/TableName)

(s/def :cognitect.aws.dynamodb.ExpectedAttributeValue/Value :cognitect.aws.dynamodb/AttributeValue)

(s/def :cognitect.aws.dynamodb.ExpectedAttributeValue/Exists :cognitect.aws.dynamodb/BooleanObject)

(s/def
  :cognitect.aws.dynamodb.ExpectedAttributeValue/ComparisonOperator
  :cognitect.aws.dynamodb/ComparisonOperator)

(s/def
  :cognitect.aws.dynamodb.ExpectedAttributeValue/AttributeValueList
  :cognitect.aws.dynamodb/AttributeValueList)

(s/def
  :cognitect.aws.dynamodb.KinesisDataStreamDestination/StreamArn
  :cognitect.aws.dynamodb/StreamArn)

(s/def
  :cognitect.aws.dynamodb.KinesisDataStreamDestination/DestinationStatus
  :cognitect.aws.dynamodb/DestinationStatus)

(s/def
  :cognitect.aws.dynamodb.KinesisDataStreamDestination/DestinationStatusDescription
  :cognitect.aws.dynamodb/String)

(s/def
  :cognitect.aws.dynamodb.ProvisionedThroughputDescription/LastIncreaseDateTime
  :cognitect.aws.dynamodb/Date)

(s/def
  :cognitect.aws.dynamodb.ProvisionedThroughputDescription/LastDecreaseDateTime
  :cognitect.aws.dynamodb/Date)

(s/def
  :cognitect.aws.dynamodb.ProvisionedThroughputDescription/NumberOfDecreasesToday
  :cognitect.aws.dynamodb/PositiveLongObject)

(s/def
  :cognitect.aws.dynamodb.ProvisionedThroughputDescription/ReadCapacityUnits
  :cognitect.aws.dynamodb/NonNegativeLongObject)

(s/def
  :cognitect.aws.dynamodb.ProvisionedThroughputDescription/WriteCapacityUnits
  :cognitect.aws.dynamodb/NonNegativeLongObject)

(s/def
  :cognitect.aws.dynamodb.ExecuteTransactionInput/TransactStatements
  :cognitect.aws.dynamodb/ParameterizedStatements)

(s/def
  :cognitect.aws.dynamodb.ExecuteTransactionInput/ClientRequestToken
  :cognitect.aws.dynamodb/ClientRequestToken)

(s/def
  :cognitect.aws.dynamodb.ExecuteTransactionInput/ReturnConsumedCapacity
  :cognitect.aws.dynamodb/ReturnConsumedCapacity)

(s/def :cognitect.aws.dynamodb.TableClassSummary/TableClass :cognitect.aws.dynamodb/TableClass)

(s/def :cognitect.aws.dynamodb.TableClassSummary/LastUpdateDateTime :cognitect.aws.dynamodb/Date)

(s/def :cognitect.aws.dynamodb.FailureException/ExceptionName :cognitect.aws.dynamodb/ExceptionName)

(s/def
  :cognitect.aws.dynamodb.FailureException/ExceptionDescription
  :cognitect.aws.dynamodb/ExceptionDescription)

(s/def :cognitect.aws.dynamodb.DescribeTableInput/TableName :cognitect.aws.dynamodb/TableName)

(s/def :cognitect.aws.dynamodb.DescribeTableOutput/Table :cognitect.aws.dynamodb/TableDescription)

(s/def :cognitect.aws.dynamodb.RestoreSummary/SourceBackupArn :cognitect.aws.dynamodb/BackupArn)

(s/def :cognitect.aws.dynamodb.RestoreSummary/SourceTableArn :cognitect.aws.dynamodb/TableArn)

(s/def :cognitect.aws.dynamodb.RestoreSummary/RestoreDateTime :cognitect.aws.dynamodb/Date)

(s/def
  :cognitect.aws.dynamodb.RestoreSummary/RestoreInProgress
  :cognitect.aws.dynamodb/RestoreInProgress)

(s/def
  :cognitect.aws.dynamodb.BackupSummary/BackupSizeBytes
  :cognitect.aws.dynamodb/BackupSizeBytes)

(s/def :cognitect.aws.dynamodb.BackupSummary/BackupExpiryDateTime :cognitect.aws.dynamodb/Date)

(s/def :cognitect.aws.dynamodb.BackupSummary/BackupStatus :cognitect.aws.dynamodb/BackupStatus)

(s/def :cognitect.aws.dynamodb.BackupSummary/TableId :cognitect.aws.dynamodb/TableId)

(s/def :cognitect.aws.dynamodb.BackupSummary/BackupType :cognitect.aws.dynamodb/BackupType)

(s/def
  :cognitect.aws.dynamodb.BackupSummary/BackupCreationDateTime
  :cognitect.aws.dynamodb/BackupCreationDateTime)

(s/def :cognitect.aws.dynamodb.BackupSummary/BackupName :cognitect.aws.dynamodb/BackupName)

(s/def :cognitect.aws.dynamodb.BackupSummary/TableName :cognitect.aws.dynamodb/TableName)

(s/def :cognitect.aws.dynamodb.BackupSummary/TableArn :cognitect.aws.dynamodb/TableArn)

(s/def :cognitect.aws.dynamodb.BackupSummary/BackupArn :cognitect.aws.dynamodb/BackupArn)

(s/def
  :cognitect.aws.dynamodb.DescribeGlobalTableInput/GlobalTableName
  :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.BatchExecuteStatementInput/Statements
  :cognitect.aws.dynamodb/PartiQLBatchRequest)

(s/def
  :cognitect.aws.dynamodb.BatchExecuteStatementInput/ReturnConsumedCapacity
  :cognitect.aws.dynamodb/ReturnConsumedCapacity)

(s/def
  :cognitect.aws.dynamodb.GlobalSecondaryIndexAutoScalingUpdate/IndexName
  :cognitect.aws.dynamodb/IndexName)

(s/def
  :cognitect.aws.dynamodb.GlobalSecondaryIndexAutoScalingUpdate/ProvisionedWriteCapacityAutoScalingUpdate
  :cognitect.aws.dynamodb/AutoScalingSettingsUpdate)

(s/def
  :cognitect.aws.dynamodb.BatchGetItemOutput/Responses
  :cognitect.aws.dynamodb/BatchGetResponseMap)

(s/def
  :cognitect.aws.dynamodb.BatchGetItemOutput/UnprocessedKeys
  :cognitect.aws.dynamodb/BatchGetRequestMap)

(s/def
  :cognitect.aws.dynamodb.BatchGetItemOutput/ConsumedCapacity
  :cognitect.aws.dynamodb/ConsumedCapacityMultiple)

(s/def :cognitect.aws.dynamodb.DeleteTableInput/TableName :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndexDescription/IndexName
  :cognitect.aws.dynamodb/IndexName)

(s/def
  :cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndexDescription/ProvisionedThroughputOverride
  :cognitect.aws.dynamodb/ProvisionedThroughputOverride)

(s/def :cognitect.aws.dynamodb.DescribeBackupInput/BackupArn :cognitect.aws.dynamodb/BackupArn)

(s/def
  :cognitect.aws.dynamodb.ContributorInsightsSummary/TableName
  :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.ContributorInsightsSummary/IndexName
  :cognitect.aws.dynamodb/IndexName)

(s/def
  :cognitect.aws.dynamodb.ContributorInsightsSummary/ContributorInsightsStatus
  :cognitect.aws.dynamodb/ContributorInsightsStatus)

(s/def
  :cognitect.aws.dynamodb.QueryInput/ReturnConsumedCapacity
  :cognitect.aws.dynamodb/ReturnConsumedCapacity)

(s/def :cognitect.aws.dynamodb.QueryInput/AttributesToGet :cognitect.aws.dynamodb/AttributeNameList)

(s/def
  :cognitect.aws.dynamodb.QueryInput/FilterExpression
  :cognitect.aws.dynamodb/ConditionExpression)

(s/def :cognitect.aws.dynamodb.QueryInput/QueryFilter :cognitect.aws.dynamodb/FilterConditionMap)

(s/def
  :cognitect.aws.dynamodb.QueryInput/ExpressionAttributeNames
  :cognitect.aws.dynamodb/ExpressionAttributeNameMap)

(s/def
  :cognitect.aws.dynamodb.QueryInput/KeyConditionExpression
  :cognitect.aws.dynamodb/KeyExpression)

(s/def :cognitect.aws.dynamodb.QueryInput/ExclusiveStartKey :cognitect.aws.dynamodb/Key)

(s/def :cognitect.aws.dynamodb.QueryInput/Limit :cognitect.aws.dynamodb/PositiveIntegerObject)

(s/def
  :cognitect.aws.dynamodb.QueryInput/ExpressionAttributeValues
  :cognitect.aws.dynamodb/ExpressionAttributeValueMap)

(s/def
  :cognitect.aws.dynamodb.QueryInput/ProjectionExpression
  :cognitect.aws.dynamodb/ProjectionExpression)

(s/def :cognitect.aws.dynamodb.QueryInput/ConsistentRead :cognitect.aws.dynamodb/ConsistentRead)

(s/def :cognitect.aws.dynamodb.QueryInput/TableName :cognitect.aws.dynamodb/TableName)

(s/def :cognitect.aws.dynamodb.QueryInput/ScanIndexForward :cognitect.aws.dynamodb/BooleanObject)

(s/def :cognitect.aws.dynamodb.QueryInput/Select :cognitect.aws.dynamodb/Select)

(s/def :cognitect.aws.dynamodb.QueryInput/IndexName :cognitect.aws.dynamodb/IndexName)

(s/def
  :cognitect.aws.dynamodb.QueryInput/ConditionalOperator
  :cognitect.aws.dynamodb/ConditionalOperator)

(s/def :cognitect.aws.dynamodb.QueryInput/KeyConditions :cognitect.aws.dynamodb/KeyConditions)

(s/def
  :cognitect.aws.dynamodb.ProvisionedThroughputOverride/ReadCapacityUnits
  :cognitect.aws.dynamodb/PositiveLongObject)

(s/def
  :cognitect.aws.dynamodb.AutoScalingTargetTrackingScalingPolicyConfigurationDescription/DisableScaleIn
  :cognitect.aws.dynamodb/BooleanObject)

(s/def
  :cognitect.aws.dynamodb.AutoScalingTargetTrackingScalingPolicyConfigurationDescription/ScaleInCooldown
  :cognitect.aws.dynamodb/IntegerObject)

(s/def
  :cognitect.aws.dynamodb.AutoScalingTargetTrackingScalingPolicyConfigurationDescription/ScaleOutCooldown
  :cognitect.aws.dynamodb/IntegerObject)

(s/def
  :cognitect.aws.dynamodb.AutoScalingTargetTrackingScalingPolicyConfigurationDescription/TargetValue
  :cognitect.aws.dynamodb/DoubleObject)

(s/def
  :cognitect.aws.dynamodb.BatchGetItemInput/RequestItems
  :cognitect.aws.dynamodb/BatchGetRequestMap)

(s/def
  :cognitect.aws.dynamodb.BatchGetItemInput/ReturnConsumedCapacity
  :cognitect.aws.dynamodb/ReturnConsumedCapacity)

(s/def
  :cognitect.aws.dynamodb.ImportTableOutput/ImportTableDescription
  :cognitect.aws.dynamodb/ImportTableDescription)

(s/def
  :cognitect.aws.dynamodb.ExportTableToPointInTimeInput/S3BucketOwner
  :cognitect.aws.dynamodb/S3BucketOwner)

(s/def
  :cognitect.aws.dynamodb.ExportTableToPointInTimeInput/S3Prefix
  :cognitect.aws.dynamodb/S3Prefix)

(s/def
  :cognitect.aws.dynamodb.ExportTableToPointInTimeInput/ExportFormat
  :cognitect.aws.dynamodb/ExportFormat)

(s/def
  :cognitect.aws.dynamodb.ExportTableToPointInTimeInput/S3Bucket
  :cognitect.aws.dynamodb/S3Bucket)

(s/def
  :cognitect.aws.dynamodb.ExportTableToPointInTimeInput/S3SseAlgorithm
  :cognitect.aws.dynamodb/S3SseAlgorithm)

(s/def
  :cognitect.aws.dynamodb.ExportTableToPointInTimeInput/ExportTime
  :cognitect.aws.dynamodb/ExportTime)

(s/def
  :cognitect.aws.dynamodb.ExportTableToPointInTimeInput/S3SseKmsKeyId
  :cognitect.aws.dynamodb/S3SseKmsKeyId)

(s/def
  :cognitect.aws.dynamodb.ExportTableToPointInTimeInput/ClientToken
  :cognitect.aws.dynamodb/ClientToken)

(s/def
  :cognitect.aws.dynamodb.ExportTableToPointInTimeInput/TableArn
  :cognitect.aws.dynamodb/TableArn)

(s/def
  :cognitect.aws.dynamodb.RestoreTableToPointInTimeOutput/TableDescription
  :cognitect.aws.dynamodb/TableDescription)

(s/def
  :cognitect.aws.dynamodb.UpdateTimeToLiveOutput/TimeToLiveSpecification
  :cognitect.aws.dynamodb/TimeToLiveSpecification)

(s/def :cognitect.aws.dynamodb.ExportDescription/EndTime :cognitect.aws.dynamodb/ExportEndTime)

(s/def
  :cognitect.aws.dynamodb.ExportDescription/BilledSizeBytes
  :cognitect.aws.dynamodb/BilledSizeBytes)

(s/def
  :cognitect.aws.dynamodb.ExportDescription/S3BucketOwner
  :cognitect.aws.dynamodb/S3BucketOwner)

(s/def :cognitect.aws.dynamodb.ExportDescription/S3Prefix :cognitect.aws.dynamodb/S3Prefix)

(s/def
  :cognitect.aws.dynamodb.ExportDescription/FailureMessage
  :cognitect.aws.dynamodb/FailureMessage)

(s/def :cognitect.aws.dynamodb.ExportDescription/ItemCount :cognitect.aws.dynamodb/ItemCount)

(s/def :cognitect.aws.dynamodb.ExportDescription/ExportFormat :cognitect.aws.dynamodb/ExportFormat)

(s/def :cognitect.aws.dynamodb.ExportDescription/S3Bucket :cognitect.aws.dynamodb/S3Bucket)

(s/def
  :cognitect.aws.dynamodb.ExportDescription/S3SseAlgorithm
  :cognitect.aws.dynamodb/S3SseAlgorithm)

(s/def :cognitect.aws.dynamodb.ExportDescription/ExportTime :cognitect.aws.dynamodb/ExportTime)

(s/def :cognitect.aws.dynamodb.ExportDescription/ExportStatus :cognitect.aws.dynamodb/ExportStatus)

(s/def :cognitect.aws.dynamodb.ExportDescription/TableId :cognitect.aws.dynamodb/TableId)

(s/def :cognitect.aws.dynamodb.ExportDescription/StartTime :cognitect.aws.dynamodb/ExportStartTime)

(s/def :cognitect.aws.dynamodb.ExportDescription/FailureCode :cognitect.aws.dynamodb/FailureCode)

(s/def
  :cognitect.aws.dynamodb.ExportDescription/S3SseKmsKeyId
  :cognitect.aws.dynamodb/S3SseKmsKeyId)

(s/def
  :cognitect.aws.dynamodb.ExportDescription/ExportManifest
  :cognitect.aws.dynamodb/ExportManifest)

(s/def :cognitect.aws.dynamodb.ExportDescription/ClientToken :cognitect.aws.dynamodb/ClientToken)

(s/def :cognitect.aws.dynamodb.ExportDescription/ExportArn :cognitect.aws.dynamodb/ExportArn)

(s/def :cognitect.aws.dynamodb.ExportDescription/TableArn :cognitect.aws.dynamodb/TableArn)

(s/def
  :cognitect.aws.dynamodb.DescribeGlobalTableSettingsOutput/GlobalTableName
  :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.DescribeGlobalTableSettingsOutput/ReplicaSettings
  :cognitect.aws.dynamodb/ReplicaSettingsDescriptionList)

(s/def
  :cognitect.aws.dynamodb.TagResourceInput/ResourceArn
  :cognitect.aws.dynamodb/ResourceArnString)

(s/def :cognitect.aws.dynamodb.TagResourceInput/Tags :cognitect.aws.dynamodb/TagList)

(s/def
  :cognitect.aws.dynamodb.BatchWriteItemInput/RequestItems
  :cognitect.aws.dynamodb/BatchWriteItemRequestMap)

(s/def
  :cognitect.aws.dynamodb.BatchWriteItemInput/ReturnConsumedCapacity
  :cognitect.aws.dynamodb/ReturnConsumedCapacity)

(s/def
  :cognitect.aws.dynamodb.BatchWriteItemInput/ReturnItemCollectionMetrics
  :cognitect.aws.dynamodb/ReturnItemCollectionMetrics)

(s/def
  :cognitect.aws.dynamodb.ExportTableToPointInTimeOutput/ExportDescription
  :cognitect.aws.dynamodb/ExportDescription)

(s/def :cognitect.aws.dynamodb.UpdateTimeToLiveInput/TableName :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.UpdateTimeToLiveInput/TimeToLiveSpecification
  :cognitect.aws.dynamodb/TimeToLiveSpecification)

(s/def :cognitect.aws.dynamodb.WriteRequest/PutRequest :cognitect.aws.dynamodb/PutRequest)

(s/def :cognitect.aws.dynamodb.WriteRequest/DeleteRequest :cognitect.aws.dynamodb/DeleteRequest)

(s/def
  :cognitect.aws.dynamodb.TransactWriteItemsInput/TransactItems
  :cognitect.aws.dynamodb/TransactWriteItemList)

(s/def
  :cognitect.aws.dynamodb.TransactWriteItemsInput/ReturnConsumedCapacity
  :cognitect.aws.dynamodb/ReturnConsumedCapacity)

(s/def
  :cognitect.aws.dynamodb.TransactWriteItemsInput/ReturnItemCollectionMetrics
  :cognitect.aws.dynamodb/ReturnItemCollectionMetrics)

(s/def
  :cognitect.aws.dynamodb.TransactWriteItemsInput/ClientRequestToken
  :cognitect.aws.dynamodb/ClientRequestToken)

(s/def :cognitect.aws.dynamodb.ExecuteStatementOutput/Items :cognitect.aws.dynamodb/ItemList)

(s/def
  :cognitect.aws.dynamodb.ExecuteStatementOutput/NextToken
  :cognitect.aws.dynamodb/PartiQLNextToken)

(s/def
  :cognitect.aws.dynamodb.ExecuteStatementOutput/ConsumedCapacity
  :cognitect.aws.dynamodb/ConsumedCapacity)

(s/def :cognitect.aws.dynamodb.ExecuteStatementOutput/LastEvaluatedKey :cognitect.aws.dynamodb/Key)

(s/def :cognitect.aws.dynamodb.ImportTableInput/ClientToken :cognitect.aws.dynamodb/ClientToken)

(s/def
  :cognitect.aws.dynamodb.ImportTableInput/S3BucketSource
  :cognitect.aws.dynamodb/S3BucketSource)

(s/def :cognitect.aws.dynamodb.ImportTableInput/InputFormat :cognitect.aws.dynamodb/InputFormat)

(s/def
  :cognitect.aws.dynamodb.ImportTableInput/InputFormatOptions
  :cognitect.aws.dynamodb/InputFormatOptions)

(s/def
  :cognitect.aws.dynamodb.ImportTableInput/InputCompressionType
  :cognitect.aws.dynamodb/InputCompressionType)

(s/def
  :cognitect.aws.dynamodb.ImportTableInput/TableCreationParameters
  :cognitect.aws.dynamodb/TableCreationParameters)

(s/def :cognitect.aws.dynamodb.ListImportsInput/TableArn :cognitect.aws.dynamodb/TableArn)

(s/def
  :cognitect.aws.dynamodb.ListImportsInput/PageSize
  :cognitect.aws.dynamodb/ListImportsMaxLimit)

(s/def :cognitect.aws.dynamodb.ListImportsInput/NextToken :cognitect.aws.dynamodb/ImportNextToken)

(s/def
  :cognitect.aws.dynamodb.UpdateReplicationGroupMemberAction/RegionName
  :cognitect.aws.dynamodb/RegionName)

(s/def
  :cognitect.aws.dynamodb.UpdateReplicationGroupMemberAction/KMSMasterKeyId
  :cognitect.aws.dynamodb/KMSMasterKeyId)

(s/def
  :cognitect.aws.dynamodb.UpdateReplicationGroupMemberAction/ProvisionedThroughputOverride
  :cognitect.aws.dynamodb/ProvisionedThroughputOverride)

(s/def
  :cognitect.aws.dynamodb.UpdateReplicationGroupMemberAction/GlobalSecondaryIndexes
  :cognitect.aws.dynamodb/ReplicaGlobalSecondaryIndexList)

(s/def
  :cognitect.aws.dynamodb.UpdateReplicationGroupMemberAction/TableClassOverride
  :cognitect.aws.dynamodb/TableClass)

(s/def :cognitect.aws.dynamodb.LocalSecondaryIndexInfo/IndexName :cognitect.aws.dynamodb/IndexName)

(s/def :cognitect.aws.dynamodb.LocalSecondaryIndexInfo/KeySchema :cognitect.aws.dynamodb/KeySchema)

(s/def
  :cognitect.aws.dynamodb.LocalSecondaryIndexInfo/Projection
  :cognitect.aws.dynamodb/Projection)

(s/def
  :cognitect.aws.dynamodb.ReplicationGroupUpdate/Create
  :cognitect.aws.dynamodb/CreateReplicationGroupMemberAction)

(s/def
  :cognitect.aws.dynamodb.ReplicationGroupUpdate/Update
  :cognitect.aws.dynamodb/UpdateReplicationGroupMemberAction)

(s/def
  :cognitect.aws.dynamodb.ReplicationGroupUpdate/Delete
  :cognitect.aws.dynamodb/DeleteReplicationGroupMemberAction)

(s/def
  :cognitect.aws.dynamodb.UpdateTableOutput/TableDescription
  :cognitect.aws.dynamodb/TableDescription)

(s/def
  :cognitect.aws.dynamodb.PutItemInput/ReturnItemCollectionMetrics
  :cognitect.aws.dynamodb/ReturnItemCollectionMetrics)

(s/def
  :cognitect.aws.dynamodb.PutItemInput/ReturnConsumedCapacity
  :cognitect.aws.dynamodb/ReturnConsumedCapacity)

(s/def
  :cognitect.aws.dynamodb.PutItemInput/ExpressionAttributeNames
  :cognitect.aws.dynamodb/ExpressionAttributeNameMap)

(s/def
  :cognitect.aws.dynamodb.PutItemInput/ConditionExpression
  :cognitect.aws.dynamodb/ConditionExpression)

(s/def
  :cognitect.aws.dynamodb.PutItemInput/ExpressionAttributeValues
  :cognitect.aws.dynamodb/ExpressionAttributeValueMap)

(s/def :cognitect.aws.dynamodb.PutItemInput/Expected :cognitect.aws.dynamodb/ExpectedAttributeMap)

(s/def :cognitect.aws.dynamodb.PutItemInput/TableName :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.PutItemInput/ReturnValuesOnConditionCheckFailure
  :cognitect.aws.dynamodb/ReturnValuesOnConditionCheckFailure)

(s/def :cognitect.aws.dynamodb.PutItemInput/ReturnValues :cognitect.aws.dynamodb/ReturnValue)

(s/def :cognitect.aws.dynamodb.PutItemInput/Item :cognitect.aws.dynamodb/PutItemInputAttributeMap)

(s/def
  :cognitect.aws.dynamodb.PutItemInput/ConditionalOperator
  :cognitect.aws.dynamodb/ConditionalOperator)

(s/def
  :cognitect.aws.dynamodb.DeleteItemInput/ReturnItemCollectionMetrics
  :cognitect.aws.dynamodb/ReturnItemCollectionMetrics)

(s/def
  :cognitect.aws.dynamodb.DeleteItemInput/ReturnConsumedCapacity
  :cognitect.aws.dynamodb/ReturnConsumedCapacity)

(s/def :cognitect.aws.dynamodb.DeleteItemInput/Key :cognitect.aws.dynamodb/Key)

(s/def
  :cognitect.aws.dynamodb.DeleteItemInput/ExpressionAttributeNames
  :cognitect.aws.dynamodb/ExpressionAttributeNameMap)

(s/def
  :cognitect.aws.dynamodb.DeleteItemInput/ConditionExpression
  :cognitect.aws.dynamodb/ConditionExpression)

(s/def
  :cognitect.aws.dynamodb.DeleteItemInput/ExpressionAttributeValues
  :cognitect.aws.dynamodb/ExpressionAttributeValueMap)

(s/def
  :cognitect.aws.dynamodb.DeleteItemInput/Expected
  :cognitect.aws.dynamodb/ExpectedAttributeMap)

(s/def :cognitect.aws.dynamodb.DeleteItemInput/TableName :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.DeleteItemInput/ReturnValuesOnConditionCheckFailure
  :cognitect.aws.dynamodb/ReturnValuesOnConditionCheckFailure)

(s/def :cognitect.aws.dynamodb.DeleteItemInput/ReturnValues :cognitect.aws.dynamodb/ReturnValue)

(s/def
  :cognitect.aws.dynamodb.DeleteItemInput/ConditionalOperator
  :cognitect.aws.dynamodb/ConditionalOperator)

(s/def
  :cognitect.aws.dynamodb.UpdateGlobalTableOutput/GlobalTableDescription
  :cognitect.aws.dynamodb/GlobalTableDescription)

(s/def
  :cognitect.aws.dynamodb.UpdateGlobalSecondaryIndexAction/IndexName
  :cognitect.aws.dynamodb/IndexName)

(s/def
  :cognitect.aws.dynamodb.UpdateGlobalSecondaryIndexAction/ProvisionedThroughput
  :cognitect.aws.dynamodb/ProvisionedThroughput)

(s/def
  :cognitect.aws.dynamodb.GlobalTableGlobalSecondaryIndexSettingsUpdate/IndexName
  :cognitect.aws.dynamodb/IndexName)

(s/def
  :cognitect.aws.dynamodb.GlobalTableGlobalSecondaryIndexSettingsUpdate/ProvisionedWriteCapacityUnits
  :cognitect.aws.dynamodb/PositiveLongObject)

(s/def
  :cognitect.aws.dynamodb.GlobalTableGlobalSecondaryIndexSettingsUpdate/ProvisionedWriteCapacityAutoScalingSettingsUpdate
  :cognitect.aws.dynamodb/AutoScalingSettingsUpdate)

(s/def
  :cognitect.aws.dynamodb.AutoScalingPolicyUpdate/PolicyName
  :cognitect.aws.dynamodb/AutoScalingPolicyName)

(s/def
  :cognitect.aws.dynamodb.AutoScalingPolicyUpdate/TargetTrackingScalingPolicyConfiguration
  :cognitect.aws.dynamodb/AutoScalingTargetTrackingScalingPolicyConfigurationUpdate)

(s/def
  :cognitect.aws.dynamodb.ScanInput/ReturnConsumedCapacity
  :cognitect.aws.dynamodb/ReturnConsumedCapacity)

(s/def :cognitect.aws.dynamodb.ScanInput/AttributesToGet :cognitect.aws.dynamodb/AttributeNameList)

(s/def
  :cognitect.aws.dynamodb.ScanInput/FilterExpression
  :cognitect.aws.dynamodb/ConditionExpression)

(s/def
  :cognitect.aws.dynamodb.ScanInput/ExpressionAttributeNames
  :cognitect.aws.dynamodb/ExpressionAttributeNameMap)

(s/def :cognitect.aws.dynamodb.ScanInput/ExclusiveStartKey :cognitect.aws.dynamodb/Key)

(s/def :cognitect.aws.dynamodb.ScanInput/Limit :cognitect.aws.dynamodb/PositiveIntegerObject)

(s/def
  :cognitect.aws.dynamodb.ScanInput/ExpressionAttributeValues
  :cognitect.aws.dynamodb/ExpressionAttributeValueMap)

(s/def
  :cognitect.aws.dynamodb.ScanInput/ProjectionExpression
  :cognitect.aws.dynamodb/ProjectionExpression)

(s/def :cognitect.aws.dynamodb.ScanInput/Segment :cognitect.aws.dynamodb/ScanSegment)

(s/def :cognitect.aws.dynamodb.ScanInput/ConsistentRead :cognitect.aws.dynamodb/ConsistentRead)

(s/def :cognitect.aws.dynamodb.ScanInput/TotalSegments :cognitect.aws.dynamodb/ScanTotalSegments)

(s/def :cognitect.aws.dynamodb.ScanInput/TableName :cognitect.aws.dynamodb/TableName)

(s/def :cognitect.aws.dynamodb.ScanInput/Select :cognitect.aws.dynamodb/Select)

(s/def :cognitect.aws.dynamodb.ScanInput/IndexName :cognitect.aws.dynamodb/IndexName)

(s/def
  :cognitect.aws.dynamodb.ScanInput/ConditionalOperator
  :cognitect.aws.dynamodb/ConditionalOperator)

(s/def :cognitect.aws.dynamodb.ScanInput/ScanFilter :cognitect.aws.dynamodb/FilterConditionMap)

(s/def :cognitect.aws.dynamodb.DescribeImportInput/ImportArn :cognitect.aws.dynamodb/ImportArn)

(s/def :cognitect.aws.dynamodb.CreateBackupInput/TableName :cognitect.aws.dynamodb/TableName)

(s/def :cognitect.aws.dynamodb.CreateBackupInput/BackupName :cognitect.aws.dynamodb/BackupName)

(s/def
  :cognitect.aws.dynamodb.CreateReplicationGroupMemberAction/RegionName
  :cognitect.aws.dynamodb/RegionName)

(s/def
  :cognitect.aws.dynamodb.CreateReplicationGroupMemberAction/KMSMasterKeyId
  :cognitect.aws.dynamodb/KMSMasterKeyId)

(s/def
  :cognitect.aws.dynamodb.CreateReplicationGroupMemberAction/ProvisionedThroughputOverride
  :cognitect.aws.dynamodb/ProvisionedThroughputOverride)

(s/def
  :cognitect.aws.dynamodb.CreateReplicationGroupMemberAction/GlobalSecondaryIndexes
  :cognitect.aws.dynamodb/ReplicaGlobalSecondaryIndexList)

(s/def
  :cognitect.aws.dynamodb.CreateReplicationGroupMemberAction/TableClassOverride
  :cognitect.aws.dynamodb/TableClass)

(s/def
  :cognitect.aws.dynamodb.DescribeKinesisStreamingDestinationInput/TableName
  :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.BackupDescription/BackupDetails
  :cognitect.aws.dynamodb/BackupDetails)

(s/def
  :cognitect.aws.dynamodb.BackupDescription/SourceTableDetails
  :cognitect.aws.dynamodb/SourceTableDetails)

(s/def
  :cognitect.aws.dynamodb.BackupDescription/SourceTableFeatureDetails
  :cognitect.aws.dynamodb/SourceTableFeatureDetails)

(s/def
  :cognitect.aws.dynamodb.StreamSpecification/StreamEnabled
  :cognitect.aws.dynamodb/StreamEnabled)

(s/def
  :cognitect.aws.dynamodb.StreamSpecification/StreamViewType
  :cognitect.aws.dynamodb/StreamViewType)

(s/def
  :cognitect.aws.dynamodb.DeleteTableOutput/TableDescription
  :cognitect.aws.dynamodb/TableDescription)

(s/def
  :cognitect.aws.dynamodb.UpdateTableReplicaAutoScalingOutput/TableAutoScalingDescription
  :cognitect.aws.dynamodb/TableAutoScalingDescription)

(s/def :cognitect.aws.dynamodb.InputFormatOptions/Csv :cognitect.aws.dynamodb/CsvOptions)

(s/def
  :cognitect.aws.dynamodb.RestoreTableFromBackupOutput/TableDescription
  :cognitect.aws.dynamodb/TableDescription)

(s/def
  :cognitect.aws.dynamodb.GlobalSecondaryIndexDescription/IndexSizeBytes
  :cognitect.aws.dynamodb/LongObject)

(s/def
  :cognitect.aws.dynamodb.GlobalSecondaryIndexDescription/IndexArn
  :cognitect.aws.dynamodb/String)

(s/def
  :cognitect.aws.dynamodb.GlobalSecondaryIndexDescription/ItemCount
  :cognitect.aws.dynamodb/LongObject)

(s/def
  :cognitect.aws.dynamodb.GlobalSecondaryIndexDescription/KeySchema
  :cognitect.aws.dynamodb/KeySchema)

(s/def
  :cognitect.aws.dynamodb.GlobalSecondaryIndexDescription/ProvisionedThroughput
  :cognitect.aws.dynamodb/ProvisionedThroughputDescription)

(s/def
  :cognitect.aws.dynamodb.GlobalSecondaryIndexDescription/Backfilling
  :cognitect.aws.dynamodb/Backfilling)

(s/def
  :cognitect.aws.dynamodb.GlobalSecondaryIndexDescription/IndexStatus
  :cognitect.aws.dynamodb/IndexStatus)

(s/def
  :cognitect.aws.dynamodb.GlobalSecondaryIndexDescription/Projection
  :cognitect.aws.dynamodb/Projection)

(s/def
  :cognitect.aws.dynamodb.GlobalSecondaryIndexDescription/IndexName
  :cognitect.aws.dynamodb/IndexName)

(s/def
  :cognitect.aws.dynamodb.DeleteBackupOutput/BackupDescription
  :cognitect.aws.dynamodb/BackupDescription)

(s/def
  :cognitect.aws.dynamodb.DescribeLimitsOutput/AccountMaxReadCapacityUnits
  :cognitect.aws.dynamodb/PositiveLongObject)

(s/def
  :cognitect.aws.dynamodb.DescribeLimitsOutput/AccountMaxWriteCapacityUnits
  :cognitect.aws.dynamodb/PositiveLongObject)

(s/def
  :cognitect.aws.dynamodb.DescribeLimitsOutput/TableMaxReadCapacityUnits
  :cognitect.aws.dynamodb/PositiveLongObject)

(s/def
  :cognitect.aws.dynamodb.DescribeLimitsOutput/TableMaxWriteCapacityUnits
  :cognitect.aws.dynamodb/PositiveLongObject)

(s/def
  :cognitect.aws.dynamodb.LocalSecondaryIndexDescription/IndexName
  :cognitect.aws.dynamodb/IndexName)

(s/def
  :cognitect.aws.dynamodb.LocalSecondaryIndexDescription/KeySchema
  :cognitect.aws.dynamodb/KeySchema)

(s/def
  :cognitect.aws.dynamodb.LocalSecondaryIndexDescription/Projection
  :cognitect.aws.dynamodb/Projection)

(s/def
  :cognitect.aws.dynamodb.LocalSecondaryIndexDescription/IndexSizeBytes
  :cognitect.aws.dynamodb/LongObject)

(s/def
  :cognitect.aws.dynamodb.LocalSecondaryIndexDescription/ItemCount
  :cognitect.aws.dynamodb/LongObject)

(s/def
  :cognitect.aws.dynamodb.LocalSecondaryIndexDescription/IndexArn
  :cognitect.aws.dynamodb/String)

(s/def
  :cognitect.aws.dynamodb.TransactWriteItem/ConditionCheck
  :cognitect.aws.dynamodb/ConditionCheck)

(s/def :cognitect.aws.dynamodb.TransactWriteItem/Put :cognitect.aws.dynamodb/Put)

(s/def :cognitect.aws.dynamodb.TransactWriteItem/Delete :cognitect.aws.dynamodb/Delete)

(s/def :cognitect.aws.dynamodb.TransactWriteItem/Update :cognitect.aws.dynamodb/Update)

(s/def
  :cognitect.aws.dynamodb.SourceTableFeatureDetails/LocalSecondaryIndexes
  :cognitect.aws.dynamodb/LocalSecondaryIndexes)

(s/def
  :cognitect.aws.dynamodb.SourceTableFeatureDetails/GlobalSecondaryIndexes
  :cognitect.aws.dynamodb/GlobalSecondaryIndexes)

(s/def
  :cognitect.aws.dynamodb.SourceTableFeatureDetails/StreamDescription
  :cognitect.aws.dynamodb/StreamSpecification)

(s/def
  :cognitect.aws.dynamodb.SourceTableFeatureDetails/TimeToLiveDescription
  :cognitect.aws.dynamodb/TimeToLiveDescription)

(s/def
  :cognitect.aws.dynamodb.SourceTableFeatureDetails/SSEDescription
  :cognitect.aws.dynamodb/SSEDescription)

(s/def
  :cognitect.aws.dynamodb.TransactGetItemsOutput/ConsumedCapacity
  :cognitect.aws.dynamodb/ConsumedCapacityMultiple)

(s/def
  :cognitect.aws.dynamodb.TransactGetItemsOutput/Responses
  :cognitect.aws.dynamodb/ItemResponseList)

(s/def
  :cognitect.aws.dynamodb.ExecuteStatementInput/Statement
  :cognitect.aws.dynamodb/PartiQLStatement)

(s/def
  :cognitect.aws.dynamodb.ExecuteStatementInput/Parameters
  :cognitect.aws.dynamodb/PreparedStatementParameters)

(s/def
  :cognitect.aws.dynamodb.ExecuteStatementInput/ConsistentRead
  :cognitect.aws.dynamodb/ConsistentRead)

(s/def
  :cognitect.aws.dynamodb.ExecuteStatementInput/NextToken
  :cognitect.aws.dynamodb/PartiQLNextToken)

(s/def
  :cognitect.aws.dynamodb.ExecuteStatementInput/ReturnConsumedCapacity
  :cognitect.aws.dynamodb/ReturnConsumedCapacity)

(s/def
  :cognitect.aws.dynamodb.ExecuteStatementInput/Limit
  :cognitect.aws.dynamodb/PositiveIntegerObject)

(s/def
  :cognitect.aws.dynamodb.ExecuteStatementInput/ReturnValuesOnConditionCheckFailure
  :cognitect.aws.dynamodb/ReturnValuesOnConditionCheckFailure)

(s/def :cognitect.aws.dynamodb.Put/Item :cognitect.aws.dynamodb/PutItemInputAttributeMap)

(s/def :cognitect.aws.dynamodb.Put/TableName :cognitect.aws.dynamodb/TableName)

(s/def :cognitect.aws.dynamodb.Put/ConditionExpression :cognitect.aws.dynamodb/ConditionExpression)

(s/def
  :cognitect.aws.dynamodb.Put/ExpressionAttributeNames
  :cognitect.aws.dynamodb/ExpressionAttributeNameMap)

(s/def
  :cognitect.aws.dynamodb.Put/ExpressionAttributeValues
  :cognitect.aws.dynamodb/ExpressionAttributeValueMap)

(s/def
  :cognitect.aws.dynamodb.Put/ReturnValuesOnConditionCheckFailure
  :cognitect.aws.dynamodb/ReturnValuesOnConditionCheckFailure)

(s/def
  :cognitect.aws.dynamodb.ListTablesInput/ExclusiveStartTableName
  :cognitect.aws.dynamodb/TableName)

(s/def :cognitect.aws.dynamodb.ListTablesInput/Limit :cognitect.aws.dynamodb/ListTablesInputLimit)

(s/def
  :cognitect.aws.dynamodb.TransactGetItemsInput/TransactItems
  :cognitect.aws.dynamodb/TransactGetItemList)

(s/def
  :cognitect.aws.dynamodb.TransactGetItemsInput/ReturnConsumedCapacity
  :cognitect.aws.dynamodb/ReturnConsumedCapacity)

(s/def
  :cognitect.aws.dynamodb.AutoScalingPolicyDescription/PolicyName
  :cognitect.aws.dynamodb/AutoScalingPolicyName)

(s/def
  :cognitect.aws.dynamodb.AutoScalingPolicyDescription/TargetTrackingScalingPolicyConfiguration
  :cognitect.aws.dynamodb/AutoScalingTargetTrackingScalingPolicyConfigurationDescription)

(s/def
  :cognitect.aws.dynamodb.UpdateTableInput/SSESpecification
  :cognitect.aws.dynamodb/SSESpecification)

(s/def :cognitect.aws.dynamodb.UpdateTableInput/BillingMode :cognitect.aws.dynamodb/BillingMode)

(s/def :cognitect.aws.dynamodb.UpdateTableInput/TableClass :cognitect.aws.dynamodb/TableClass)

(s/def
  :cognitect.aws.dynamodb.UpdateTableInput/GlobalSecondaryIndexUpdates
  :cognitect.aws.dynamodb/GlobalSecondaryIndexUpdateList)

(s/def
  :cognitect.aws.dynamodb.UpdateTableInput/ReplicaUpdates
  :cognitect.aws.dynamodb/ReplicationGroupUpdateList)

(s/def
  :cognitect.aws.dynamodb.UpdateTableInput/StreamSpecification
  :cognitect.aws.dynamodb/StreamSpecification)

(s/def
  :cognitect.aws.dynamodb.UpdateTableInput/ProvisionedThroughput
  :cognitect.aws.dynamodb/ProvisionedThroughput)

(s/def :cognitect.aws.dynamodb.UpdateTableInput/TableName :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.UpdateTableInput/AttributeDefinitions
  :cognitect.aws.dynamodb/AttributeDefinitions)

(s/def
  :cognitect.aws.dynamodb.UpdateTableInput/DeletionProtectionEnabled
  :cognitect.aws.dynamodb/DeletionProtectionEnabled)

(s/def :cognitect.aws.dynamodb.CreateReplicaAction/RegionName :cognitect.aws.dynamodb/RegionName)

(s/def
  :cognitect.aws.dynamodb.KinesisStreamingDestinationOutput/TableName
  :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.KinesisStreamingDestinationOutput/StreamArn
  :cognitect.aws.dynamodb/StreamArn)

(s/def
  :cognitect.aws.dynamodb.KinesisStreamingDestinationOutput/DestinationStatus
  :cognitect.aws.dynamodb/DestinationStatus)

(s/def :cognitect.aws.dynamodb.DeleteRequest/Key :cognitect.aws.dynamodb/Key)

(s/def
  :cognitect.aws.dynamodb.BatchStatementError/Code
  :cognitect.aws.dynamodb/BatchStatementErrorCodeEnum)

(s/def :cognitect.aws.dynamodb.BatchStatementError/Message :cognitect.aws.dynamodb/String)

(s/def :cognitect.aws.dynamodb.BatchStatementError/Item :cognitect.aws.dynamodb/AttributeMap)

(s/def
  :cognitect.aws.dynamodb.UntagResourceInput/ResourceArn
  :cognitect.aws.dynamodb/ResourceArnString)

(s/def :cognitect.aws.dynamodb.UntagResourceInput/TagKeys :cognitect.aws.dynamodb/TagKeyList)

(s/def
  :cognitect.aws.dynamodb.TransactWriteItemsOutput/ConsumedCapacity
  :cognitect.aws.dynamodb/ConsumedCapacityMultiple)

(s/def
  :cognitect.aws.dynamodb.TransactWriteItemsOutput/ItemCollectionMetrics
  :cognitect.aws.dynamodb/ItemCollectionMetricsPerTable)

(s/def :cognitect.aws.dynamodb.BackupDetails/BackupArn :cognitect.aws.dynamodb/BackupArn)

(s/def :cognitect.aws.dynamodb.BackupDetails/BackupName :cognitect.aws.dynamodb/BackupName)

(s/def
  :cognitect.aws.dynamodb.BackupDetails/BackupSizeBytes
  :cognitect.aws.dynamodb/BackupSizeBytes)

(s/def :cognitect.aws.dynamodb.BackupDetails/BackupStatus :cognitect.aws.dynamodb/BackupStatus)

(s/def :cognitect.aws.dynamodb.BackupDetails/BackupType :cognitect.aws.dynamodb/BackupType)

(s/def
  :cognitect.aws.dynamodb.BackupDetails/BackupCreationDateTime
  :cognitect.aws.dynamodb/BackupCreationDateTime)

(s/def :cognitect.aws.dynamodb.BackupDetails/BackupExpiryDateTime :cognitect.aws.dynamodb/Date)

(s/def
  :cognitect.aws.dynamodb.RestoreTableToPointInTimeInput/BillingModeOverride
  :cognitect.aws.dynamodb/BillingMode)

(s/def
  :cognitect.aws.dynamodb.RestoreTableToPointInTimeInput/ProvisionedThroughputOverride
  :cognitect.aws.dynamodb/ProvisionedThroughput)

(s/def
  :cognitect.aws.dynamodb.RestoreTableToPointInTimeInput/TargetTableName
  :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.RestoreTableToPointInTimeInput/LocalSecondaryIndexOverride
  :cognitect.aws.dynamodb/LocalSecondaryIndexList)

(s/def
  :cognitect.aws.dynamodb.RestoreTableToPointInTimeInput/UseLatestRestorableTime
  :cognitect.aws.dynamodb/BooleanObject)

(s/def
  :cognitect.aws.dynamodb.RestoreTableToPointInTimeInput/SourceTableArn
  :cognitect.aws.dynamodb/TableArn)

(s/def
  :cognitect.aws.dynamodb.RestoreTableToPointInTimeInput/SourceTableName
  :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.RestoreTableToPointInTimeInput/SSESpecificationOverride
  :cognitect.aws.dynamodb/SSESpecification)

(s/def
  :cognitect.aws.dynamodb.RestoreTableToPointInTimeInput/GlobalSecondaryIndexOverride
  :cognitect.aws.dynamodb/GlobalSecondaryIndexList)

(s/def
  :cognitect.aws.dynamodb.RestoreTableToPointInTimeInput/RestoreDateTime
  :cognitect.aws.dynamodb/Date)

(s/def
  :cognitect.aws.dynamodb.ProvisionedThroughput/ReadCapacityUnits
  :cognitect.aws.dynamodb/PositiveLongObject)

(s/def
  :cognitect.aws.dynamodb.ProvisionedThroughput/WriteCapacityUnits
  :cognitect.aws.dynamodb/PositiveLongObject)

(s/def
  :cognitect.aws.dynamodb.DescribeTimeToLiveOutput/TimeToLiveDescription
  :cognitect.aws.dynamodb/TimeToLiveDescription)

(s/def :cognitect.aws.dynamodb.Delete/Key :cognitect.aws.dynamodb/Key)

(s/def :cognitect.aws.dynamodb.Delete/TableName :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.Delete/ConditionExpression
  :cognitect.aws.dynamodb/ConditionExpression)

(s/def
  :cognitect.aws.dynamodb.Delete/ExpressionAttributeNames
  :cognitect.aws.dynamodb/ExpressionAttributeNameMap)

(s/def
  :cognitect.aws.dynamodb.Delete/ExpressionAttributeValues
  :cognitect.aws.dynamodb/ExpressionAttributeValueMap)

(s/def
  :cognitect.aws.dynamodb.Delete/ReturnValuesOnConditionCheckFailure
  :cognitect.aws.dynamodb/ReturnValuesOnConditionCheckFailure)

(s/def :cognitect.aws.dynamodb.SourceTableDetails/BillingMode :cognitect.aws.dynamodb/BillingMode)

(s/def :cognitect.aws.dynamodb.SourceTableDetails/ItemCount :cognitect.aws.dynamodb/ItemCount)

(s/def :cognitect.aws.dynamodb.SourceTableDetails/TableId :cognitect.aws.dynamodb/TableId)

(s/def
  :cognitect.aws.dynamodb.SourceTableDetails/TableCreationDateTime
  :cognitect.aws.dynamodb/TableCreationDateTime)

(s/def :cognitect.aws.dynamodb.SourceTableDetails/KeySchema :cognitect.aws.dynamodb/KeySchema)

(s/def
  :cognitect.aws.dynamodb.SourceTableDetails/ProvisionedThroughput
  :cognitect.aws.dynamodb/ProvisionedThroughput)

(s/def :cognitect.aws.dynamodb.SourceTableDetails/TableName :cognitect.aws.dynamodb/TableName)

(s/def :cognitect.aws.dynamodb.SourceTableDetails/TableArn :cognitect.aws.dynamodb/TableArn)

(s/def :cognitect.aws.dynamodb.SourceTableDetails/TableSizeBytes :cognitect.aws.dynamodb/LongObject)

(s/def
  :cognitect.aws.dynamodb.DescribeEndpointsResponse/Endpoints
  :cognitect.aws.dynamodb/Endpoints)

(s/def
  :cognitect.aws.dynamodb.CreateGlobalSecondaryIndexAction/IndexName
  :cognitect.aws.dynamodb/IndexName)

(s/def
  :cognitect.aws.dynamodb.CreateGlobalSecondaryIndexAction/KeySchema
  :cognitect.aws.dynamodb/KeySchema)

(s/def
  :cognitect.aws.dynamodb.CreateGlobalSecondaryIndexAction/Projection
  :cognitect.aws.dynamodb/Projection)

(s/def
  :cognitect.aws.dynamodb.CreateGlobalSecondaryIndexAction/ProvisionedThroughput
  :cognitect.aws.dynamodb/ProvisionedThroughput)

(s/def :cognitect.aws.dynamodb.AttributeValueUpdate/Value :cognitect.aws.dynamodb/AttributeValue)

(s/def :cognitect.aws.dynamodb.AttributeValueUpdate/Action :cognitect.aws.dynamodb/AttributeAction)

(s/def
  :cognitect.aws.dynamodb.ExecuteTransactionOutput/Responses
  :cognitect.aws.dynamodb/ItemResponseList)

(s/def
  :cognitect.aws.dynamodb.ExecuteTransactionOutput/ConsumedCapacity
  :cognitect.aws.dynamodb/ConsumedCapacityMultiple)

(s/def :cognitect.aws.dynamodb.TableDescription/LatestStreamArn :cognitect.aws.dynamodb/StreamArn)

(s/def :cognitect.aws.dynamodb.TableDescription/TableStatus :cognitect.aws.dynamodb/TableStatus)

(s/def :cognitect.aws.dynamodb.TableDescription/LatestStreamLabel :cognitect.aws.dynamodb/String)

(s/def
  :cognitect.aws.dynamodb.TableDescription/TableClassSummary
  :cognitect.aws.dynamodb/TableClassSummary)

(s/def
  :cognitect.aws.dynamodb.TableDescription/RestoreSummary
  :cognitect.aws.dynamodb/RestoreSummary)

(s/def :cognitect.aws.dynamodb.TableDescription/ItemCount :cognitect.aws.dynamodb/LongObject)

(s/def
  :cognitect.aws.dynamodb.TableDescription/StreamSpecification
  :cognitect.aws.dynamodb/StreamSpecification)

(s/def :cognitect.aws.dynamodb.TableDescription/TableId :cognitect.aws.dynamodb/TableId)

(s/def
  :cognitect.aws.dynamodb.TableDescription/GlobalSecondaryIndexes
  :cognitect.aws.dynamodb/GlobalSecondaryIndexDescriptionList)

(s/def :cognitect.aws.dynamodb.TableDescription/KeySchema :cognitect.aws.dynamodb/KeySchema)

(s/def
  :cognitect.aws.dynamodb.TableDescription/ProvisionedThroughput
  :cognitect.aws.dynamodb/ProvisionedThroughputDescription)

(s/def :cognitect.aws.dynamodb.TableDescription/CreationDateTime :cognitect.aws.dynamodb/Date)

(s/def
  :cognitect.aws.dynamodb.TableDescription/LocalSecondaryIndexes
  :cognitect.aws.dynamodb/LocalSecondaryIndexDescriptionList)

(s/def
  :cognitect.aws.dynamodb.TableDescription/ArchivalSummary
  :cognitect.aws.dynamodb/ArchivalSummary)

(s/def :cognitect.aws.dynamodb.TableDescription/GlobalTableVersion :cognitect.aws.dynamodb/String)

(s/def :cognitect.aws.dynamodb.TableDescription/TableName :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.TableDescription/SSEDescription
  :cognitect.aws.dynamodb/SSEDescription)

(s/def
  :cognitect.aws.dynamodb.TableDescription/Replicas
  :cognitect.aws.dynamodb/ReplicaDescriptionList)

(s/def
  :cognitect.aws.dynamodb.TableDescription/AttributeDefinitions
  :cognitect.aws.dynamodb/AttributeDefinitions)

(s/def
  :cognitect.aws.dynamodb.TableDescription/DeletionProtectionEnabled
  :cognitect.aws.dynamodb/DeletionProtectionEnabled)

(s/def :cognitect.aws.dynamodb.TableDescription/TableArn :cognitect.aws.dynamodb/String)

(s/def :cognitect.aws.dynamodb.TableDescription/TableSizeBytes :cognitect.aws.dynamodb/LongObject)

(s/def
  :cognitect.aws.dynamodb.TableDescription/BillingModeSummary
  :cognitect.aws.dynamodb/BillingModeSummary)

(s/def
  :cognitect.aws.dynamodb.PointInTimeRecoverySpecification/PointInTimeRecoveryEnabled
  :cognitect.aws.dynamodb/BooleanObject)

(s/def :cognitect.aws.dynamodb.ArchivalSummary/ArchivalDateTime :cognitect.aws.dynamodb/Date)

(s/def
  :cognitect.aws.dynamodb.ArchivalSummary/ArchivalReason
  :cognitect.aws.dynamodb/ArchivalReason)

(s/def :cognitect.aws.dynamodb.ArchivalSummary/ArchivalBackupArn :cognitect.aws.dynamodb/BackupArn)

(s/def :cognitect.aws.dynamodb.Get/Key :cognitect.aws.dynamodb/Key)

(s/def :cognitect.aws.dynamodb.Get/TableName :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.Get/ProjectionExpression
  :cognitect.aws.dynamodb/ProjectionExpression)

(s/def
  :cognitect.aws.dynamodb.Get/ExpressionAttributeNames
  :cognitect.aws.dynamodb/ExpressionAttributeNameMap)

(s/def
  :cognitect.aws.dynamodb.UpdateGlobalTableInput/GlobalTableName
  :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.UpdateGlobalTableInput/ReplicaUpdates
  :cognitect.aws.dynamodb/ReplicaUpdateList)

(s/def
  :cognitect.aws.dynamodb.AutoScalingTargetTrackingScalingPolicyConfigurationUpdate/DisableScaleIn
  :cognitect.aws.dynamodb/BooleanObject)

(s/def
  :cognitect.aws.dynamodb.AutoScalingTargetTrackingScalingPolicyConfigurationUpdate/ScaleInCooldown
  :cognitect.aws.dynamodb/IntegerObject)

(s/def
  :cognitect.aws.dynamodb.AutoScalingTargetTrackingScalingPolicyConfigurationUpdate/ScaleOutCooldown
  :cognitect.aws.dynamodb/IntegerObject)

(s/def
  :cognitect.aws.dynamodb.AutoScalingTargetTrackingScalingPolicyConfigurationUpdate/TargetValue
  :cognitect.aws.dynamodb/DoubleObject)

(s/def
  :cognitect.aws.dynamodb.UpdateGlobalTableSettingsOutput/GlobalTableName
  :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.UpdateGlobalTableSettingsOutput/ReplicaSettings
  :cognitect.aws.dynamodb/ReplicaSettingsDescriptionList)

(s/def :cognitect.aws.dynamodb.PutItemOutput/Attributes :cognitect.aws.dynamodb/AttributeMap)

(s/def
  :cognitect.aws.dynamodb.PutItemOutput/ConsumedCapacity
  :cognitect.aws.dynamodb/ConsumedCapacity)

(s/def
  :cognitect.aws.dynamodb.PutItemOutput/ItemCollectionMetrics
  :cognitect.aws.dynamodb/ItemCollectionMetrics)

(s/def
  :cognitect.aws.dynamodb.CreateTableInput/SSESpecification
  :cognitect.aws.dynamodb/SSESpecification)

(s/def :cognitect.aws.dynamodb.CreateTableInput/BillingMode :cognitect.aws.dynamodb/BillingMode)

(s/def :cognitect.aws.dynamodb.CreateTableInput/TableClass :cognitect.aws.dynamodb/TableClass)

(s/def :cognitect.aws.dynamodb.CreateTableInput/Tags :cognitect.aws.dynamodb/TagList)

(s/def
  :cognitect.aws.dynamodb.CreateTableInput/StreamSpecification
  :cognitect.aws.dynamodb/StreamSpecification)

(s/def
  :cognitect.aws.dynamodb.CreateTableInput/GlobalSecondaryIndexes
  :cognitect.aws.dynamodb/GlobalSecondaryIndexList)

(s/def :cognitect.aws.dynamodb.CreateTableInput/KeySchema :cognitect.aws.dynamodb/KeySchema)

(s/def
  :cognitect.aws.dynamodb.CreateTableInput/ProvisionedThroughput
  :cognitect.aws.dynamodb/ProvisionedThroughput)

(s/def
  :cognitect.aws.dynamodb.CreateTableInput/LocalSecondaryIndexes
  :cognitect.aws.dynamodb/LocalSecondaryIndexList)

(s/def :cognitect.aws.dynamodb.CreateTableInput/TableName :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.CreateTableInput/AttributeDefinitions
  :cognitect.aws.dynamodb/AttributeDefinitions)

(s/def
  :cognitect.aws.dynamodb.CreateTableInput/DeletionProtectionEnabled
  :cognitect.aws.dynamodb/DeletionProtectionEnabled)

(s/def :cognitect.aws.dynamodb.ConditionCheck/Key :cognitect.aws.dynamodb/Key)

(s/def :cognitect.aws.dynamodb.ConditionCheck/TableName :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.ConditionCheck/ConditionExpression
  :cognitect.aws.dynamodb/ConditionExpression)

(s/def
  :cognitect.aws.dynamodb.ConditionCheck/ExpressionAttributeNames
  :cognitect.aws.dynamodb/ExpressionAttributeNameMap)

(s/def
  :cognitect.aws.dynamodb.ConditionCheck/ExpressionAttributeValues
  :cognitect.aws.dynamodb/ExpressionAttributeValueMap)

(s/def
  :cognitect.aws.dynamodb.ConditionCheck/ReturnValuesOnConditionCheckFailure
  :cognitect.aws.dynamodb/ReturnValuesOnConditionCheckFailure)

(s/def
  :cognitect.aws.dynamodb.DescribeContributorInsightsInput/TableName
  :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.DescribeContributorInsightsInput/IndexName
  :cognitect.aws.dynamodb/IndexName)

(s/def
  :cognitect.aws.dynamodb.BatchStatementRequest/Statement
  :cognitect.aws.dynamodb/PartiQLStatement)

(s/def
  :cognitect.aws.dynamodb.BatchStatementRequest/Parameters
  :cognitect.aws.dynamodb/PreparedStatementParameters)

(s/def
  :cognitect.aws.dynamodb.BatchStatementRequest/ConsistentRead
  :cognitect.aws.dynamodb/ConsistentRead)

(s/def
  :cognitect.aws.dynamodb.BatchStatementRequest/ReturnValuesOnConditionCheckFailure
  :cognitect.aws.dynamodb/ReturnValuesOnConditionCheckFailure)

(s/def
  :cognitect.aws.dynamodb.PointInTimeRecoveryDescription/PointInTimeRecoveryStatus
  :cognitect.aws.dynamodb/PointInTimeRecoveryStatus)

(s/def
  :cognitect.aws.dynamodb.PointInTimeRecoveryDescription/EarliestRestorableDateTime
  :cognitect.aws.dynamodb/Date)

(s/def
  :cognitect.aws.dynamodb.PointInTimeRecoveryDescription/LatestRestorableDateTime
  :cognitect.aws.dynamodb/Date)

(s/def
  :cognitect.aws.dynamodb.UpdateContinuousBackupsOutput/ContinuousBackupsDescription
  :cognitect.aws.dynamodb/ContinuousBackupsDescription)

(s/def
  :cognitect.aws.dynamodb.AttributeDefinition/AttributeName
  :cognitect.aws.dynamodb/KeySchemaAttributeName)

(s/def
  :cognitect.aws.dynamodb.AttributeDefinition/AttributeType
  :cognitect.aws.dynamodb/ScalarAttributeType)

(s/def :cognitect.aws.dynamodb.ConsumedCapacity/TableName :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.ConsumedCapacity/CapacityUnits
  :cognitect.aws.dynamodb/ConsumedCapacityUnits)

(s/def
  :cognitect.aws.dynamodb.ConsumedCapacity/ReadCapacityUnits
  :cognitect.aws.dynamodb/ConsumedCapacityUnits)

(s/def
  :cognitect.aws.dynamodb.ConsumedCapacity/WriteCapacityUnits
  :cognitect.aws.dynamodb/ConsumedCapacityUnits)

(s/def :cognitect.aws.dynamodb.ConsumedCapacity/Table :cognitect.aws.dynamodb/Capacity)

(s/def
  :cognitect.aws.dynamodb.ConsumedCapacity/LocalSecondaryIndexes
  :cognitect.aws.dynamodb/SecondaryIndexesCapacityMap)

(s/def
  :cognitect.aws.dynamodb.ConsumedCapacity/GlobalSecondaryIndexes
  :cognitect.aws.dynamodb/SecondaryIndexesCapacityMap)

(s/def
  :cognitect.aws.dynamodb.KinesisStreamingDestinationInput/TableName
  :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.KinesisStreamingDestinationInput/StreamArn
  :cognitect.aws.dynamodb/StreamArn)

(s/def
  :cognitect.aws.dynamodb.GlobalSecondaryIndexUpdate/Update
  :cognitect.aws.dynamodb/UpdateGlobalSecondaryIndexAction)

(s/def
  :cognitect.aws.dynamodb.GlobalSecondaryIndexUpdate/Create
  :cognitect.aws.dynamodb/CreateGlobalSecondaryIndexAction)

(s/def
  :cognitect.aws.dynamodb.GlobalSecondaryIndexUpdate/Delete
  :cognitect.aws.dynamodb/DeleteGlobalSecondaryIndexAction)

(s/def
  :cognitect.aws.dynamodb.ListExportsOutput/ExportSummaries
  :cognitect.aws.dynamodb/ExportSummaries)

(s/def :cognitect.aws.dynamodb.ListExportsOutput/NextToken :cognitect.aws.dynamodb/ExportNextToken)

(s/def :cognitect.aws.dynamodb.KeysAndAttributes/Keys :cognitect.aws.dynamodb/KeyList)

(s/def
  :cognitect.aws.dynamodb.KeysAndAttributes/AttributesToGet
  :cognitect.aws.dynamodb/AttributeNameList)

(s/def
  :cognitect.aws.dynamodb.KeysAndAttributes/ConsistentRead
  :cognitect.aws.dynamodb/ConsistentRead)

(s/def
  :cognitect.aws.dynamodb.KeysAndAttributes/ProjectionExpression
  :cognitect.aws.dynamodb/ProjectionExpression)

(s/def
  :cognitect.aws.dynamodb.KeysAndAttributes/ExpressionAttributeNames
  :cognitect.aws.dynamodb/ExpressionAttributeNameMap)

(s/def
  :cognitect.aws.dynamodb.ItemCollectionMetrics/ItemCollectionKey
  :cognitect.aws.dynamodb/ItemCollectionKeyAttributeMap)

(s/def
  :cognitect.aws.dynamodb.ItemCollectionMetrics/SizeEstimateRangeGB
  :cognitect.aws.dynamodb/ItemCollectionSizeEstimateRange)

(s/def :cognitect.aws.dynamodb.SSEDescription/Status :cognitect.aws.dynamodb/SSEStatus)

(s/def :cognitect.aws.dynamodb.SSEDescription/SSEType :cognitect.aws.dynamodb/SSEType)

(s/def
  :cognitect.aws.dynamodb.SSEDescription/KMSMasterKeyArn
  :cognitect.aws.dynamodb/KMSMasterKeyArn)

(s/def
  :cognitect.aws.dynamodb.SSEDescription/InaccessibleEncryptionDateTime
  :cognitect.aws.dynamodb/Date)

(s/def :cognitect.aws.dynamodb.GlobalSecondaryIndex/IndexName :cognitect.aws.dynamodb/IndexName)

(s/def :cognitect.aws.dynamodb.GlobalSecondaryIndex/KeySchema :cognitect.aws.dynamodb/KeySchema)

(s/def :cognitect.aws.dynamodb.GlobalSecondaryIndex/Projection :cognitect.aws.dynamodb/Projection)

(s/def
  :cognitect.aws.dynamodb.GlobalSecondaryIndex/ProvisionedThroughput
  :cognitect.aws.dynamodb/ProvisionedThroughput)

(s/def
  :cognitect.aws.dynamodb.CreateGlobalTableInput/GlobalTableName
  :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.CreateGlobalTableInput/ReplicationGroup
  :cognitect.aws.dynamodb/ReplicaList)

(s/def
  :cognitect.aws.dynamodb.DescribeGlobalTableSettingsInput/GlobalTableName
  :cognitect.aws.dynamodb/TableName)

(s/def :cognitect.aws.dynamodb.GlobalSecondaryIndexInfo/IndexName :cognitect.aws.dynamodb/IndexName)

(s/def :cognitect.aws.dynamodb.GlobalSecondaryIndexInfo/KeySchema :cognitect.aws.dynamodb/KeySchema)

(s/def
  :cognitect.aws.dynamodb.GlobalSecondaryIndexInfo/Projection
  :cognitect.aws.dynamodb/Projection)

(s/def
  :cognitect.aws.dynamodb.GlobalSecondaryIndexInfo/ProvisionedThroughput
  :cognitect.aws.dynamodb/ProvisionedThroughput)

(s/def :cognitect.aws.dynamodb.GlobalTable/GlobalTableName :cognitect.aws.dynamodb/TableName)

(s/def :cognitect.aws.dynamodb.GlobalTable/ReplicationGroup :cognitect.aws.dynamodb/ReplicaList)

(s/def :cognitect.aws.dynamodb.DeleteItemOutput/Attributes :cognitect.aws.dynamodb/AttributeMap)

(s/def
  :cognitect.aws.dynamodb.DeleteItemOutput/ConsumedCapacity
  :cognitect.aws.dynamodb/ConsumedCapacity)

(s/def
  :cognitect.aws.dynamodb.DeleteItemOutput/ItemCollectionMetrics
  :cognitect.aws.dynamodb/ItemCollectionMetrics)

(s/def
  :cognitect.aws.dynamodb.Condition/AttributeValueList
  :cognitect.aws.dynamodb/AttributeValueList)

(s/def
  :cognitect.aws.dynamodb.Condition/ComparisonOperator
  :cognitect.aws.dynamodb/ComparisonOperator)

(s/def
  :cognitect.aws.dynamodb.TimeToLiveDescription/TimeToLiveStatus
  :cognitect.aws.dynamodb/TimeToLiveStatus)

(s/def
  :cognitect.aws.dynamodb.TimeToLiveDescription/AttributeName
  :cognitect.aws.dynamodb/TimeToLiveAttributeName)

(s/def
  :cognitect.aws.dynamodb.KeySchemaElement/AttributeName
  :cognitect.aws.dynamodb/KeySchemaAttributeName)

(s/def :cognitect.aws.dynamodb.KeySchemaElement/KeyType :cognitect.aws.dynamodb/KeyType)

(s/def
  :cognitect.aws.dynamodb.DeleteGlobalSecondaryIndexAction/IndexName
  :cognitect.aws.dynamodb/IndexName)

(s/def
  :cognitect.aws.dynamodb.CreateGlobalTableOutput/GlobalTableDescription
  :cognitect.aws.dynamodb/GlobalTableDescription)

(s/def :cognitect.aws.dynamodb.ExportSummary/ExportArn :cognitect.aws.dynamodb/ExportArn)

(s/def :cognitect.aws.dynamodb.ExportSummary/ExportStatus :cognitect.aws.dynamodb/ExportStatus)

(s/def :cognitect.aws.dynamodb.Projection/ProjectionType :cognitect.aws.dynamodb/ProjectionType)

(s/def
  :cognitect.aws.dynamodb.Projection/NonKeyAttributes
  :cognitect.aws.dynamodb/NonKeyAttributeNameList)

(s/def
  :cognitect.aws.dynamodb.UpdateGlobalTableSettingsInput/GlobalTableName
  :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.UpdateGlobalTableSettingsInput/GlobalTableBillingMode
  :cognitect.aws.dynamodb/BillingMode)

(s/def
  :cognitect.aws.dynamodb.UpdateGlobalTableSettingsInput/GlobalTableProvisionedWriteCapacityUnits
  :cognitect.aws.dynamodb/PositiveLongObject)

(s/def
  :cognitect.aws.dynamodb.UpdateGlobalTableSettingsInput/GlobalTableProvisionedWriteCapacityAutoScalingSettingsUpdate
  :cognitect.aws.dynamodb/AutoScalingSettingsUpdate)

(s/def
  :cognitect.aws.dynamodb.UpdateGlobalTableSettingsInput/GlobalTableGlobalSecondaryIndexSettingsUpdate
  :cognitect.aws.dynamodb/GlobalTableGlobalSecondaryIndexSettingsUpdateList)

(s/def
  :cognitect.aws.dynamodb.UpdateGlobalTableSettingsInput/ReplicaSettingsUpdate
  :cognitect.aws.dynamodb/ReplicaSettingsUpdateList)

(s/def
  :cognitect.aws.dynamodb.DescribeContributorInsightsOutput/TableName
  :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.DescribeContributorInsightsOutput/IndexName
  :cognitect.aws.dynamodb/IndexName)

(s/def
  :cognitect.aws.dynamodb.DescribeContributorInsightsOutput/ContributorInsightsRuleList
  :cognitect.aws.dynamodb/ContributorInsightsRuleList)

(s/def
  :cognitect.aws.dynamodb.DescribeContributorInsightsOutput/ContributorInsightsStatus
  :cognitect.aws.dynamodb/ContributorInsightsStatus)

(s/def
  :cognitect.aws.dynamodb.DescribeContributorInsightsOutput/LastUpdateDateTime
  :cognitect.aws.dynamodb/LastUpdateDateTime)

(s/def
  :cognitect.aws.dynamodb.DescribeContributorInsightsOutput/FailureException
  :cognitect.aws.dynamodb/FailureException)

(s/def :cognitect.aws.dynamodb.TableCreationParameters/TableName :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.TableCreationParameters/AttributeDefinitions
  :cognitect.aws.dynamodb/AttributeDefinitions)

(s/def :cognitect.aws.dynamodb.TableCreationParameters/KeySchema :cognitect.aws.dynamodb/KeySchema)

(s/def
  :cognitect.aws.dynamodb.TableCreationParameters/BillingMode
  :cognitect.aws.dynamodb/BillingMode)

(s/def
  :cognitect.aws.dynamodb.TableCreationParameters/ProvisionedThroughput
  :cognitect.aws.dynamodb/ProvisionedThroughput)

(s/def
  :cognitect.aws.dynamodb.TableCreationParameters/SSESpecification
  :cognitect.aws.dynamodb/SSESpecification)

(s/def
  :cognitect.aws.dynamodb.TableCreationParameters/GlobalSecondaryIndexes
  :cognitect.aws.dynamodb/GlobalSecondaryIndexList)

(s/def :cognitect.aws.dynamodb.ScanOutput/Items :cognitect.aws.dynamodb/ItemList)

(s/def :cognitect.aws.dynamodb.ScanOutput/Count :cognitect.aws.dynamodb/Integer)

(s/def :cognitect.aws.dynamodb.ScanOutput/ScannedCount :cognitect.aws.dynamodb/Integer)

(s/def :cognitect.aws.dynamodb.ScanOutput/LastEvaluatedKey :cognitect.aws.dynamodb/Key)

(s/def :cognitect.aws.dynamodb.ScanOutput/ConsumedCapacity :cognitect.aws.dynamodb/ConsumedCapacity)

(s/def
  :cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndexAutoScalingUpdate/IndexName
  :cognitect.aws.dynamodb/IndexName)

(s/def
  :cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndexAutoScalingUpdate/ProvisionedReadCapacityAutoScalingUpdate
  :cognitect.aws.dynamodb/AutoScalingSettingsUpdate)

(s/def
  :cognitect.aws.dynamodb.ReplicaAutoScalingDescription/RegionName
  :cognitect.aws.dynamodb/RegionName)

(s/def
  :cognitect.aws.dynamodb.ReplicaAutoScalingDescription/GlobalSecondaryIndexes
  :cognitect.aws.dynamodb/ReplicaGlobalSecondaryIndexAutoScalingDescriptionList)

(s/def
  :cognitect.aws.dynamodb.ReplicaAutoScalingDescription/ReplicaProvisionedReadCapacityAutoScalingSettings
  :cognitect.aws.dynamodb/AutoScalingSettingsDescription)

(s/def
  :cognitect.aws.dynamodb.ReplicaAutoScalingDescription/ReplicaProvisionedWriteCapacityAutoScalingSettings
  :cognitect.aws.dynamodb/AutoScalingSettingsDescription)

(s/def
  :cognitect.aws.dynamodb.ReplicaAutoScalingDescription/ReplicaStatus
  :cognitect.aws.dynamodb/ReplicaStatus)

(s/def
  :cognitect.aws.dynamodb.ContinuousBackupsDescription/ContinuousBackupsStatus
  :cognitect.aws.dynamodb/ContinuousBackupsStatus)

(s/def
  :cognitect.aws.dynamodb.ContinuousBackupsDescription/PointInTimeRecoveryDescription
  :cognitect.aws.dynamodb/PointInTimeRecoveryDescription)

(s/def
  :cognitect.aws.dynamodb.UpdateContinuousBackupsInput/TableName
  :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.UpdateContinuousBackupsInput/PointInTimeRecoverySpecification
  :cognitect.aws.dynamodb/PointInTimeRecoverySpecification)

(s/def
  :cognitect.aws.dynamodb.Capacity/ReadCapacityUnits
  :cognitect.aws.dynamodb/ConsumedCapacityUnits)

(s/def
  :cognitect.aws.dynamodb.Capacity/WriteCapacityUnits
  :cognitect.aws.dynamodb/ConsumedCapacityUnits)

(s/def :cognitect.aws.dynamodb.Capacity/CapacityUnits :cognitect.aws.dynamodb/ConsumedCapacityUnits)

(s/def :cognitect.aws.dynamodb.GetItemOutput/Item :cognitect.aws.dynamodb/AttributeMap)

(s/def
  :cognitect.aws.dynamodb.GetItemOutput/ConsumedCapacity
  :cognitect.aws.dynamodb/ConsumedCapacity)

(s/def
  :cognitect.aws.dynamodb.DescribeExportOutput/ExportDescription
  :cognitect.aws.dynamodb/ExportDescription)

(s/def
  :cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndexAutoScalingDescription/IndexName
  :cognitect.aws.dynamodb/IndexName)

(s/def
  :cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndexAutoScalingDescription/IndexStatus
  :cognitect.aws.dynamodb/IndexStatus)

(s/def
  :cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndexAutoScalingDescription/ProvisionedReadCapacityAutoScalingSettings
  :cognitect.aws.dynamodb/AutoScalingSettingsDescription)

(s/def
  :cognitect.aws.dynamodb.ReplicaGlobalSecondaryIndexAutoScalingDescription/ProvisionedWriteCapacityAutoScalingSettings
  :cognitect.aws.dynamodb/AutoScalingSettingsDescription)

(s/def
  :cognitect.aws.dynamodb.ListGlobalTablesInput/ExclusiveStartGlobalTableName
  :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.ListGlobalTablesInput/Limit
  :cognitect.aws.dynamodb/PositiveIntegerObject)

(s/def :cognitect.aws.dynamodb.ListGlobalTablesInput/RegionName :cognitect.aws.dynamodb/RegionName)

(s/def :cognitect.aws.dynamodb.DescribeTimeToLiveInput/TableName :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.UpdateContributorInsightsInput/TableName
  :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.UpdateContributorInsightsInput/IndexName
  :cognitect.aws.dynamodb/IndexName)

(s/def
  :cognitect.aws.dynamodb.UpdateContributorInsightsInput/ContributorInsightsAction
  :cognitect.aws.dynamodb/ContributorInsightsAction)

(s/def :cognitect.aws.dynamodb.ListBackupsInput/TableName :cognitect.aws.dynamodb/TableName)

(s/def :cognitect.aws.dynamodb.ListBackupsInput/Limit :cognitect.aws.dynamodb/BackupsInputLimit)

(s/def
  :cognitect.aws.dynamodb.ListBackupsInput/TimeRangeLowerBound
  :cognitect.aws.dynamodb/TimeRangeLowerBound)

(s/def
  :cognitect.aws.dynamodb.ListBackupsInput/TimeRangeUpperBound
  :cognitect.aws.dynamodb/TimeRangeUpperBound)

(s/def
  :cognitect.aws.dynamodb.ListBackupsInput/ExclusiveStartBackupArn
  :cognitect.aws.dynamodb/BackupArn)

(s/def :cognitect.aws.dynamodb.ListBackupsInput/BackupType :cognitect.aws.dynamodb/BackupTypeFilter)

(s/def :cognitect.aws.dynamodb.AttributeValue/L :cognitect.aws.dynamodb/ListAttributeValue)

(s/def :cognitect.aws.dynamodb.AttributeValue/M :cognitect.aws.dynamodb/MapAttributeValue)

(s/def :cognitect.aws.dynamodb.AttributeValue/NS :cognitect.aws.dynamodb/NumberSetAttributeValue)

(s/def :cognitect.aws.dynamodb.AttributeValue/BOOL :cognitect.aws.dynamodb/BooleanAttributeValue)

(s/def :cognitect.aws.dynamodb.AttributeValue/B :cognitect.aws.dynamodb/BinaryAttributeValue)

(s/def :cognitect.aws.dynamodb.AttributeValue/SS :cognitect.aws.dynamodb/StringSetAttributeValue)

(s/def :cognitect.aws.dynamodb.AttributeValue/NULL :cognitect.aws.dynamodb/NullAttributeValue)

(s/def :cognitect.aws.dynamodb.AttributeValue/S :cognitect.aws.dynamodb/StringAttributeValue)

(s/def :cognitect.aws.dynamodb.AttributeValue/BS :cognitect.aws.dynamodb/BinarySetAttributeValue)

(s/def :cognitect.aws.dynamodb.AttributeValue/N :cognitect.aws.dynamodb/NumberAttributeValue)

(s/def
  :cognitect.aws.dynamodb.ListTagsOfResourceInput/ResourceArn
  :cognitect.aws.dynamodb/ResourceArnString)

(s/def
  :cognitect.aws.dynamodb.ListTagsOfResourceInput/NextToken
  :cognitect.aws.dynamodb/NextTokenString)

(s/def
  :cognitect.aws.dynamodb.UpdateItemInput/ReturnItemCollectionMetrics
  :cognitect.aws.dynamodb/ReturnItemCollectionMetrics)

(s/def
  :cognitect.aws.dynamodb.UpdateItemInput/ReturnConsumedCapacity
  :cognitect.aws.dynamodb/ReturnConsumedCapacity)

(s/def
  :cognitect.aws.dynamodb.UpdateItemInput/AttributeUpdates
  :cognitect.aws.dynamodb/AttributeUpdates)

(s/def :cognitect.aws.dynamodb.UpdateItemInput/Key :cognitect.aws.dynamodb/Key)

(s/def
  :cognitect.aws.dynamodb.UpdateItemInput/ExpressionAttributeNames
  :cognitect.aws.dynamodb/ExpressionAttributeNameMap)

(s/def
  :cognitect.aws.dynamodb.UpdateItemInput/ConditionExpression
  :cognitect.aws.dynamodb/ConditionExpression)

(s/def
  :cognitect.aws.dynamodb.UpdateItemInput/ExpressionAttributeValues
  :cognitect.aws.dynamodb/ExpressionAttributeValueMap)

(s/def
  :cognitect.aws.dynamodb.UpdateItemInput/Expected
  :cognitect.aws.dynamodb/ExpectedAttributeMap)

(s/def
  :cognitect.aws.dynamodb.UpdateItemInput/UpdateExpression
  :cognitect.aws.dynamodb/UpdateExpression)

(s/def :cognitect.aws.dynamodb.UpdateItemInput/TableName :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.UpdateItemInput/ReturnValuesOnConditionCheckFailure
  :cognitect.aws.dynamodb/ReturnValuesOnConditionCheckFailure)

(s/def :cognitect.aws.dynamodb.UpdateItemInput/ReturnValues :cognitect.aws.dynamodb/ReturnValue)

(s/def
  :cognitect.aws.dynamodb.UpdateItemInput/ConditionalOperator
  :cognitect.aws.dynamodb/ConditionalOperator)

(s/def :cognitect.aws.dynamodb.GetItemInput/TableName :cognitect.aws.dynamodb/TableName)

(s/def :cognitect.aws.dynamodb.GetItemInput/Key :cognitect.aws.dynamodb/Key)

(s/def
  :cognitect.aws.dynamodb.GetItemInput/AttributesToGet
  :cognitect.aws.dynamodb/AttributeNameList)

(s/def :cognitect.aws.dynamodb.GetItemInput/ConsistentRead :cognitect.aws.dynamodb/ConsistentRead)

(s/def
  :cognitect.aws.dynamodb.GetItemInput/ReturnConsumedCapacity
  :cognitect.aws.dynamodb/ReturnConsumedCapacity)

(s/def
  :cognitect.aws.dynamodb.GetItemInput/ProjectionExpression
  :cognitect.aws.dynamodb/ProjectionExpression)

(s/def
  :cognitect.aws.dynamodb.GetItemInput/ExpressionAttributeNames
  :cognitect.aws.dynamodb/ExpressionAttributeNameMap)

(s/def :cognitect.aws.dynamodb.Replica/RegionName :cognitect.aws.dynamodb/RegionName)

(s/def
  :cognitect.aws.dynamodb.DescribeTableReplicaAutoScalingOutput/TableAutoScalingDescription
  :cognitect.aws.dynamodb/TableAutoScalingDescription)

(s/def :cognitect.aws.dynamodb.DescribeExportInput/ExportArn :cognitect.aws.dynamodb/ExportArn)

(s/def
  :cognitect.aws.dynamodb.RestoreTableFromBackupInput/TargetTableName
  :cognitect.aws.dynamodb/TableName)

(s/def
  :cognitect.aws.dynamodb.RestoreTableFromBackupInput/BackupArn
  :cognitect.aws.dynamodb/BackupArn)

(s/def
  :cognitect.aws.dynamodb.RestoreTableFromBackupInput/BillingModeOverride
  :cognitect.aws.dynamodb/BillingMode)

(s/def
  :cognitect.aws.dynamodb.RestoreTableFromBackupInput/GlobalSecondaryIndexOverride
  :cognitect.aws.dynamodb/GlobalSecondaryIndexList)

(s/def
  :cognitect.aws.dynamodb.RestoreTableFromBackupInput/LocalSecondaryIndexOverride
  :cognitect.aws.dynamodb/LocalSecondaryIndexList)

(s/def
  :cognitect.aws.dynamodb.RestoreTableFromBackupInput/ProvisionedThroughputOverride
  :cognitect.aws.dynamodb/ProvisionedThroughput)

(s/def
  :cognitect.aws.dynamodb.RestoreTableFromBackupInput/SSESpecificationOverride
  :cognitect.aws.dynamodb/SSESpecification)

(s/def :cognitect.aws.dynamodb.DeleteReplicaAction/RegionName :cognitect.aws.dynamodb/RegionName)

(s/def :cognitect.aws.dynamodb.BillingModeSummary/BillingMode :cognitect.aws.dynamodb/BillingMode)

(s/def
  :cognitect.aws.dynamodb.BillingModeSummary/LastUpdateToPayPerRequestDateTime
  :cognitect.aws.dynamodb/Date)

(s/def
  :cognitect.aws.dynamodb.TimeToLiveSpecification/Enabled
  :cognitect.aws.dynamodb/TimeToLiveEnabled)

(s/def
  :cognitect.aws.dynamodb.TimeToLiveSpecification/AttributeName
  :cognitect.aws.dynamodb/TimeToLiveAttributeName)

(s/def :cognitect.aws.dynamodb.ItemResponse/Item :cognitect.aws.dynamodb/AttributeMap)

