/*
 * Decompiled with CFR 0.152.
 */
package io.prestosql.operator;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import io.airlift.slice.SizeOf;
import io.prestosql.array.IntBigArray;
import io.prestosql.operator.IdRegistry;
import io.prestosql.spi.Page;
import io.prestosql.util.LongBigArrayFIFOQueue;
import it.unimi.dsi.fastutil.ints.IntIterator;
import it.unimi.dsi.fastutil.ints.IntOpenHashSet;
import it.unimi.dsi.fastutil.longs.LongComparator;
import java.util.Arrays;
import javax.annotation.Nullable;
import org.openjdk.jol.info.ClassLayout;

public class RowReferencePageManager {
    private static final long INSTANCE_SIZE = ClassLayout.parseClass(RowReferencePageManager.class).instanceSize();
    private static final long PAGE_ACCOUNTING_INSTANCE_SIZE = ClassLayout.parseClass(PageAccounting.class).instanceSize();
    private static final int RESERVED_ROW_ID_FOR_CURSOR = -1;
    private final IdRegistry<PageAccounting> pages = new IdRegistry();
    private final RowIdBuffer rowIdBuffer = new RowIdBuffer();
    private final IntHashSet compactionCandidates = new IntHashSet();
    @Nullable
    private LoadCursor currentCursor;
    private long pageBytes;

    public LoadCursor add(Page page) {
        Preconditions.checkState((this.currentCursor == null ? 1 : 0) != 0, (Object)"Cursor still active");
        int pageId = this.pages.allocateId(id -> new PageAccounting(id, page));
        PageAccounting pageAccounting = this.pages.get(pageId);
        pageAccounting.lockPage();
        this.currentCursor = new LoadCursor(pageAccounting, () -> {
            Preconditions.checkState((this.currentCursor != null ? 1 : 0) != 0);
            pageAccounting.unlockPage();
            pageAccounting.loadPageLoadIfNeeded();
            this.pageBytes += pageAccounting.sizeOf();
            this.currentCursor = null;
            this.checkPageMaintenance(pageAccounting);
        });
        return this.currentCursor;
    }

    public void dereference(long rowId) {
        PageAccounting pageAccounting = this.pages.get(this.rowIdBuffer.getPageId(rowId));
        pageAccounting.dereference(rowId);
        this.checkPageMaintenance(pageAccounting);
    }

    private void checkPageMaintenance(PageAccounting pageAccounting) {
        int pageId = pageAccounting.getPageId();
        if (pageAccounting.isPruneEligible()) {
            this.compactionCandidates.remove(pageId);
            this.pages.deallocate(pageId);
            this.pageBytes -= pageAccounting.sizeOf();
        } else if (pageAccounting.isCompactionEligible()) {
            this.compactionCandidates.add(pageId);
        }
    }

    public Page getPage(long rowId) {
        if (RowReferencePageManager.isCursorRowId(rowId)) {
            Preconditions.checkState((this.currentCursor != null ? 1 : 0) != 0, (Object)"No active cursor");
            return this.currentCursor.getPage();
        }
        int pageId = this.rowIdBuffer.getPageId(rowId);
        return this.pages.get(pageId).getPage();
    }

    public int getPosition(long rowId) {
        if (RowReferencePageManager.isCursorRowId(rowId)) {
            Preconditions.checkState((this.currentCursor != null ? 1 : 0) != 0, (Object)"No active cursor");
            return this.currentCursor.getCurrentPosition();
        }
        return this.rowIdBuffer.getPosition(rowId);
    }

    private static boolean isCursorRowId(long rowId) {
        return rowId == -1L;
    }

    public void compactIfNeeded() {
        IntIterator iterator = this.compactionCandidates.iterator();
        while (iterator.hasNext()) {
            int pageId = iterator.nextInt();
            PageAccounting pageAccounting = this.pages.get(pageId);
            this.pageBytes -= pageAccounting.sizeOf();
            pageAccounting.compact();
            this.pageBytes += pageAccounting.sizeOf();
        }
        this.compactionCandidates.clear();
    }

    @VisibleForTesting
    int getCompactionCandidateCount() {
        return this.compactionCandidates.size();
    }

    @VisibleForTesting
    long getPageBytes() {
        return this.pageBytes;
    }

    public long sizeOf() {
        return INSTANCE_SIZE + this.pageBytes + this.pages.sizeOf() + this.rowIdBuffer.sizeOf() + this.compactionCandidates.sizeOf();
    }

    private static class IntHashSet
    extends IntOpenHashSet {
        private static final long INSTANCE_SIZE = ClassLayout.parseClass(IntHashSet.class).instanceSize();

        private IntHashSet() {
        }

        public long sizeOf() {
            return INSTANCE_SIZE + SizeOf.sizeOf((int[])this.key);
        }
    }

    private static class RowIdBuffer {
        public static final long UNKNOWN_ID = -1L;
        private static final long INSTANCE_SIZE = ClassLayout.parseClass(RowIdBuffer.class).instanceSize();
        private final IntBigArray buffer = new IntBigArray();
        private final LongBigArrayFIFOQueue emptySlots = new LongBigArrayFIFOQueue();
        private long capacity;

        private RowIdBuffer() {
        }

        public long allocateRowId(int pageId, int position) {
            long newRowId;
            if (!this.emptySlots.isEmpty()) {
                newRowId = this.emptySlots.dequeueLong();
            } else {
                newRowId = this.capacity++;
                this.buffer.ensureCapacity(this.capacity * 2L);
            }
            this.setPageId(newRowId, pageId);
            this.setPosition(newRowId, position);
            return newRowId;
        }

        public void deallocate(long rowId) {
            this.emptySlots.enqueue(rowId);
        }

        public int getPageId(long rowId) {
            return this.buffer.get(rowId * 2L);
        }

        public void setPageId(long rowId, int pageId) {
            this.buffer.set(rowId * 2L, pageId);
        }

        public int getPosition(long rowId) {
            return this.buffer.get(rowId * 2L + 1L);
        }

        public void setPosition(long rowId, int position) {
            this.buffer.set(rowId * 2L + 1L, position);
        }

        public long sizeOf() {
            return INSTANCE_SIZE + this.buffer.sizeOf() + this.emptySlots.sizeOf();
        }
    }

    private class PageAccounting {
        private static final int COMPACTION_MIN_FILL_MULTIPLIER = 2;
        private final int pageId;
        private Page page;
        private boolean isPageLoaded;
        private long[] rowIds;
        private boolean lockedPage = true;
        private int activePositions;

        public PageAccounting(int pageId, Page page) {
            this.pageId = pageId;
            this.page = page;
            this.rowIds = new long[page.getPositionCount()];
            Arrays.fill(this.rowIds, -1L);
        }

        public long referencePosition(int position) {
            long rowId = this.rowIds[position];
            if (rowId == -1L) {
                this.rowIds[position] = rowId = RowReferencePageManager.this.rowIdBuffer.allocateRowId(this.pageId, position);
                ++this.activePositions;
            }
            return rowId;
        }

        public void lockPage() {
            this.lockedPage = true;
        }

        public void unlockPage() {
            this.lockedPage = false;
        }

        public int getPageId() {
            return this.pageId;
        }

        public Page getPage() {
            return this.page;
        }

        public void dereference(long rowId) {
            int position = RowReferencePageManager.this.rowIdBuffer.getPosition(rowId);
            Preconditions.checkArgument((rowId == this.rowIds[position] ? 1 : 0) != 0, (Object)"rowId does not match this page");
            this.rowIds[position] = -1L;
            --this.activePositions;
            RowReferencePageManager.this.rowIdBuffer.deallocate(rowId);
        }

        public boolean isPruneEligible() {
            return !this.lockedPage && this.activePositions == 0;
        }

        public boolean isCompactionEligible() {
            return !this.lockedPage && this.activePositions * 2 < this.page.getPositionCount();
        }

        public void loadPageLoadIfNeeded() {
            if (!this.isPageLoaded && this.activePositions > 0) {
                this.page = this.page.getLoadedPage();
                this.isPageLoaded = true;
            }
        }

        public void compact() {
            Preconditions.checkState((!this.lockedPage ? 1 : 0) != 0, (Object)"Should not attempt compaction when page is locked");
            if (this.activePositions == this.page.getPositionCount()) {
                return;
            }
            this.loadPageLoadIfNeeded();
            int newIndex = 0;
            int[] positionsToKeep = new int[this.activePositions];
            long[] newRowIds = new long[this.activePositions];
            for (int i = 0; i < this.page.getPositionCount(); ++i) {
                long rowId = this.rowIds[i];
                if (rowId == -1L) continue;
                positionsToKeep[newIndex] = i;
                newRowIds[newIndex] = rowId;
                RowReferencePageManager.this.rowIdBuffer.setPosition(rowId, newIndex);
                ++newIndex;
            }
            Verify.verify((newIndex == this.activePositions ? 1 : 0) != 0);
            this.page = this.page.copyPositions(positionsToKeep, 0, this.activePositions);
            this.rowIds = newRowIds;
        }

        public long sizeOf() {
            long loadedPageSize = this.isPageLoaded ? this.page.getSizeInBytes() : 0L;
            return PAGE_ACCOUNTING_INSTANCE_SIZE + loadedPageSize + SizeOf.sizeOf((long[])this.rowIds);
        }
    }

    public static class LoadCursor
    implements AutoCloseable {
        private final PageAccounting pageAccounting;
        private final Runnable closeCallback;
        private int currentPosition = -1;

        private LoadCursor(PageAccounting pageAccounting, Runnable closeCallback) {
            this.pageAccounting = pageAccounting;
            this.closeCallback = closeCallback;
        }

        private Page getPage() {
            return this.pageAccounting.getPage();
        }

        private int getCurrentPosition() {
            Preconditions.checkState((this.currentPosition >= 0 ? 1 : 0) != 0, (Object)"Not yet advanced");
            return this.currentPosition;
        }

        public boolean advance() {
            if (this.currentPosition >= this.pageAccounting.getPage().getPositionCount() - 1) {
                return false;
            }
            ++this.currentPosition;
            return true;
        }

        public int compareTo(LongComparator rowIdComparator, long rowId) {
            Preconditions.checkState((this.currentPosition >= 0 ? 1 : 0) != 0, (Object)"Not yet advanced");
            return rowIdComparator.compare(-1L, rowId);
        }

        public long allocateRowId() {
            Preconditions.checkState((this.currentPosition >= 0 ? 1 : 0) != 0, (Object)"Not yet advanced");
            return this.pageAccounting.referencePosition(this.currentPosition);
        }

        @Override
        public void close() {
            this.closeCallback.run();
        }
    }
}

