/*
 * Decompiled with CFR 0.152.
 */
package io.prestosql.server.security.oauth2;

import com.google.common.base.Strings;
import com.google.common.base.Verify;
import com.google.common.collect.Ordering;
import com.google.common.hash.Hashing;
import com.google.common.io.Resources;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.Jws;
import io.jsonwebtoken.JwtParser;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.SignatureAlgorithm;
import io.jsonwebtoken.SigningKeyResolver;
import io.prestosql.server.security.oauth2.ChallengeFailedException;
import io.prestosql.server.security.oauth2.ForOAuth2;
import io.prestosql.server.security.oauth2.OAuth2Client;
import io.prestosql.server.security.oauth2.OAuth2Config;
import java.io.IOException;
import java.net.URI;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.security.SecureRandom;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.TemporalAmount;
import java.util.Date;
import java.util.Objects;
import javax.inject.Inject;

public class OAuth2Service {
    private static final String STATE_AUDIENCE = "presto_oauth";
    private static final String FAILURE_REPLACEMENT_TEXT = "<!-- ERROR_MESSAGE -->";
    private final OAuth2Client client;
    private final JwtParser jwtParser;
    private final String failureHtml;
    private final TemporalAmount challengeTimeout;
    private final byte[] stateHmac;

    @Inject
    public OAuth2Service(OAuth2Client client, @ForOAuth2 SigningKeyResolver signingKeyResolver, OAuth2Config oauth2Config) throws IOException {
        this.client = Objects.requireNonNull(client, "client is null");
        this.jwtParser = Jwts.parser().setSigningKeyResolver(signingKeyResolver);
        this.failureHtml = Resources.toString((URL)Resources.getResource(this.getClass(), (String)"/oauth2/failure.html"), (Charset)StandardCharsets.UTF_8);
        Verify.verify((boolean)this.failureHtml.contains(FAILURE_REPLACEMENT_TEXT), (String)"login.html does not contain the replacement text", (Object[])new Object[0]);
        Objects.requireNonNull(oauth2Config, "oauth2Config is null");
        this.challengeTimeout = Duration.ofMillis(oauth2Config.getChallengeTimeout().toMillis());
        this.stateHmac = oauth2Config.getStateKey().map(key -> Hashing.sha256().hashString((CharSequence)key, StandardCharsets.UTF_8).asBytes()).orElseGet(() -> OAuth2Service.secureRandomBytes(32));
    }

    public URI startChallenge(URI callbackUri) {
        String state = Jwts.builder().signWith(SignatureAlgorithm.HS256, this.stateHmac).setAudience(STATE_AUDIENCE).setExpiration(Date.from(Instant.now().plus(this.challengeTimeout))).compact();
        return this.client.getAuthorizationUri(state, callbackUri);
    }

    public OAuthResult finishChallenge(String state, String code, URI callbackUri) throws ChallengeFailedException {
        Objects.requireNonNull(callbackUri, "callbackUri is null");
        Objects.requireNonNull(state, "state is null");
        Objects.requireNonNull(code, "code is null");
        Claims stateClaims = this.parseState(state);
        if (!STATE_AUDIENCE.equals(stateClaims.getAudience())) {
            throw new ChallengeFailedException(String.format("Unexpected state audience: %s. Expected audience: %s.", stateClaims.getAudience(), STATE_AUDIENCE));
        }
        OAuth2Client.AccessToken accessToken = this.client.getAccessToken(code, callbackUri);
        Claims parsedToken = (Claims)this.jwtParser.parseClaimsJws(accessToken.getAccessToken()).getBody();
        Instant validUntil = accessToken.getValidUntil().map(instant -> (Instant)Ordering.natural().min(instant, (Object)parsedToken.getExpiration().toInstant())).orElse(parsedToken.getExpiration().toInstant());
        return new OAuthResult(accessToken.getAccessToken(), validUntil);
    }

    private Claims parseState(String state) throws ChallengeFailedException {
        try {
            return (Claims)Jwts.parser().setSigningKey(this.stateHmac).parseClaimsJws(state).getBody();
        }
        catch (RuntimeException e) {
            throw new ChallengeFailedException("State validation failed", e);
        }
    }

    public Jws<Claims> parseClaimsJws(String token) {
        return this.jwtParser.parseClaimsJws(token);
    }

    public String getCallbackErrorHtml(String errorCode) {
        return this.failureHtml.replace(FAILURE_REPLACEMENT_TEXT, OAuth2Service.getOAuth2ErrorMessage(errorCode));
    }

    public String getInternalFailureHtml(String errorMessage) {
        return this.failureHtml.replace(FAILURE_REPLACEMENT_TEXT, Strings.nullToEmpty((String)errorMessage));
    }

    private static byte[] secureRandomBytes(int count) {
        byte[] bytes = new byte[count];
        new SecureRandom().nextBytes(bytes);
        return bytes;
    }

    private static String getOAuth2ErrorMessage(String errorCode) {
        switch (errorCode) {
            case "access_denied": {
                return "OAuth2 server denied the login";
            }
            case "unauthorized_client": {
                return "OAuth2 server does not allow request from this Presto server";
            }
            case "server_error": {
                return "OAuth2 server had a failure";
            }
            case "temporarily_unavailable": {
                return "OAuth2 server is temporarily unavailable";
            }
        }
        return "OAuth2 unknown error code: " + errorCode;
    }

    public static class OAuthResult {
        private final String accessToken;
        private final Instant tokenExpiration;

        public OAuthResult(String accessToken, Instant tokenExpiration) {
            this.accessToken = Objects.requireNonNull(accessToken, "accessToken is null");
            this.tokenExpiration = Objects.requireNonNull(tokenExpiration, "tokenExpiration is null");
        }

        public String getAccessToken() {
            return this.accessToken;
        }

        public Instant getTokenExpiration() {
            return this.tokenExpiration;
        }
    }
}

