;; Copyright (c) Cognitect, Inc.
;; All rights reserved.

;; Licensed under the Apache License, Version 2.0 (the "License");
;; you may not use this file except in compliance with the License.
;; You may obtain a copy of the License at
;;
;;      http://www.apache.org/licenses/LICENSE-2.0
;;
;; Unless required by applicable law or agreed to in writing, software
;; distributed under the License is distributed on an "AS-IS" BASIS,
;; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
;; See the License for the specific language governing permissions and
;; limitations under the License.

(ns cognitect.hmac-authn.specs
  (:require
   [cognitect.hmac-authn :as hmac-authn]))

(in-ns 'cognitect.hmac-authn)
(import java.nio.ByteBuffer)
(require '[clojure.spec.gen.alpha :as gen]
         '[clojure.spec.alpha :as s]
         '[cognitect.anomalies :as anom])

(defn byte-buffer?
  [x]
  (instance? ByteBuffer x))


(s/def ::non-empty-string (s/and string? #(not (empty? %))))
(s/def ::byte-buffer (s/spec byte-buffer? :gen #(gen/fmap (fn [bs] (ByteBuffer/wrap bs))
                                                          (s/gen bytes?))))
(s/def ::sha (s/spec ::non-empty-string :gen #(s/gen #{"e3b0c44298fc1c149afbf4c8996fb92427ae41e4649b934ca495991b7852b855"
                                                       "6e340b9cffb37a989ca544e6bb780a2c78901d3fb33738768511a30617afa01d"
                                                       "b413f47d13ee2fe6c845b2ee141af81de858df4ec549a58b7970bb96645bc8d2"})))

(s/def ::access-key-id ::non-empty-string)
(s/def ::body (s/or :bytes bytes?
                    :byte-buffer ::byte-buffer))
(s/def ::canonical-request-hash ::sha)
(s/def ::content-length nat-int?)
(s/def ::credential-scope ::non-empty-string)
(s/def ::headers (s/map-of ::non-empty-string string?))
(s/def ::region ::non-empty-string)
(s/def ::secret ::non-empty-string)
(s/def ::service ::non-empty-string)
(s/def ::string-to-sign ::non-empty-string)
(s/def ::uri ::non-empty-string)
(s/def ::x-amz-date (s/spec ::non-empty-string :gen #(s/gen #{"20170206T144605Z"
                                                             "20170401T120000Z"
                                                             "20171031T120000Z"})))
(s/def ::x-amz-content-sha256 ::sha)
(s/def ::x-amz-target (s/spec ::non-empty-string :gen #(gen/fmap str (s/gen qualified-keyword?))))

(s/def ::http-req (s/keys :req-un [::uri ::body ::content-length ::headers]))
(s/def ::signed-req (s/and ::http-req signed?))
(s/def ::sign-params-base (s/keys :req-un [::access-key-id ::region ::service]))
(s/def ::sign-params-with-secret (s/merge ::sign-params-base (s/keys :req-un [::secret])))
(s/def ::sign-params (s/spec ::sign-params-base :gen (s/gen ::sign-params-with-secret)))
(s/def ::request-date ::non-empty-string)
(s/def ::signed-headers (s/coll-of ::non-empty-string :type set?))
(s/def ::signature ::non-empty-string)
(s/def ::parsed-authorization-header (s/keys :req-un [::access-key-id ::request-date ::service ::region ::signed-headers ::signature]))

;; doesn't work for string specs
#_(s/def ::x-amz-headers (s/keys :req-un [::x-amz-date ::x-amz-content-sha256]
                               :opt=un [::x-amz-target] ))

(s/fdef sign
        :args (s/cat :req ::http-req
                     :params ::sign-params)
        :ret (s/or :signed ::signed-req
                   :failed ::anom/anomaly))

(s/fdef can-verify?
        :args (s/cat :req ::http-req
                     :params ::sign-params)
        :ret boolean?)

(s/fdef verify-failure
        :args (s/cat :req ::http-req
                     :params ::sign-params)
        :ret (s/nilable keyword?))

(s/fdef parse-authorization-header
        :args (s/cat :auth-header string?)
        :ret (s/nilable ::parsed-authorization-header))
