;; Copyright (c) Cognitect, Inc.
;; All rights reserved.
(ns datomic.s3-access-keys.specs
  (:require datomic.s3-access-keys))

(in-ns 'datomic.s3-access-keys)
(require '[clojure.spec.alpha :as s]
         '[clojure.string :as str])

(s/def ::non-empty-string (s/and string? #(not-empty %)))
(s/def ::path (s/and ::non-empty-string
                     #(not (str/starts-with? % "/"))
                     #(not (str/ends-with? % "/"))))

;; path-elem is one component of a path delimited by "/"
(s/def ::path-elem (s/and ::non-empty-string
                     #(not (str/index-of % "/"))))
(s/def ::prefix ::non-empty-string)
(s/def ::access-key-id ::non-empty-string)
(s/def ::system ::path-elem)
(s/def ::db-name ::path-elem)
(s/def ::key-name ::path-elem)
(s/def ::created inst?)
(s/def ::type keyword?)
(s/def ::secret ::non-empty-string)

(defn- kt
  "Returns a predicate that checks a key for :type type."
  [type]
  (fn [k] (= (:type k) type)))

(s/def ::access-key (s/keys :req-un [::type ::system ::key-name]))
(s/def ::db-access-key (s/merge ::access-key (s/keys :req-un [::db-name])))

(s/def ::admin-key (s/and ::access-key (kt :admin)))
(s/def ::catalog-read-key (s/and ::access-key (kt :catalog-read)))
(s/def ::db-write-key (s/and ::db-access-key (kt :db-write)))
(s/def ::db-read-key (s/and ::db-access-key (kt :db-read)))

(s/fdef parse-access-key-id
        :args (s/cat :s (s/or :valid ::access-key-id :invalid any?))
        :ret (s/nilable ::access-key))

(s/fdef create-access-key-id
        :args (s/cat :m (s/or :valid ::access-key :invalid any?))
        :ret (s/nilable ::access-key-id))


