;; Copyright (c) Cognitect, Inc.
;; All rights reserved.

(ns cognitect.s3-creds.keyfile
  (:import
   [java.security SecureRandom]
   [java.util Date]
   [org.apache.commons.codec.binary Base64 Hex])
  (:require
   [clojure.string :as str]))

(defn- string->bytes
  "Converts with UTF-8."
  [^String s]
  (.getBytes s "UTF-8"))

(defn- bytes->string
  "Converts with UTF-8."
  [^bytes b]
  (String. b "UTF-8"))

(defn- encode-64
  ^bytes [raw]
  ;(.encode (Base64.) raw)
  (Base64/encodeBase64 raw false))

(defn- decode-64
  [^bytes coded]
  (Base64/decodeBase64 coded))

(defn- encode-hex
  ^bytes [^bytes raw]
  (.encode (Hex.) raw))

(defn- random-bytes
  "Returns random bytes with at least entropy bits of entropy."
  ^bytes [entropy]
  (let [sr (SecureRandom.)
        b (byte-array (quot (+ 7 entropy) 8))]
    (.nextBytes sr b)
    b))

(defn- matching-creds
  "Returns creds if they match key-name-in."
  [{:keys [key-name] :as creds} key-name-in]
  (when (= key-name-in key-name)
    creds))

(defn- creds-for-key-name
  "Returns creds at keyname in keyfile, or nil."
  [{:keys [current previous]} key-name]
  (or (matching-creds current key-name)
      (matching-creds previous key-name)))

(def ^:private not-found {:cognitect.anomalies/category
                          :cognitect.anomalies/not-found})

(defn parse-access-key-id
  [s]
  (when-let [[_ keyfile-dir _ key-name] (re-find #"(s3://(.+))/([^/]+)" s)]
    {:keyfile-name (str keyfile-dir "/.keys")
     :key-name key-name}))

(defn create-access-key-id
  [keyfile-name key-name]
  (when-let [[_ keyfile-dir] (re-find #"(s3://(.+))/([^/]+)" keyfile-name)]
    (str keyfile-dir "/" key-name)))

(defn create-creds
  "Create new ::creds."
  []
  {:key-name (-> (random-bytes 64) encode-hex bytes->string)
   :secret (-> (random-bytes 256) encode-64 bytes->string)
   :created (Date.)})

(defn rotate-creds
  "Rotate new creds into ::keyfile structure."
  ([] (rotate-creds nil))
  ([{:keys [current]}]
     (merge {:current (create-creds)}
            (when current {:previous current}))))

(defn current-creds
  "Return the current creds for parsed access-key-id and keyfile"
  [{:keys [keyfile-name]} keyfile]
  (when-let [current (:current keyfile)]
    {:access-key-id (create-access-key-id keyfile-name (:key-name current))
     :creds current}))

(defn creds
  "Return the specific creds named by this parsed access key,
if they exist in keyfile."
  [{:keys [keyfile-name key-name]} keyfile]
  (when-let [creds (creds-for-key-name keyfile key-name)]
    {:access-key-id (create-access-key-id keyfile-name key-name)
     :creds creds}))



