;; Copyright (c) Cognitect, Inc.
;; All rights reserved.

(ns cognitect.s3-creds.specs
  (:require
   cognitect.anomalies
   cognitect.s3-creds
   cognitect.s3-creds.cache
   cognitect.s3-creds.keyfile
   cognitect.s3-creds.store))

(in-ns 'cognitect.s3-creds)
(require
 '[clojure.spec.alpha :as s]
 '[clojure.string :as str]
 '[cognitect.anomalies :as anom])

(import
  '[com.amazonaws.services.s3 AmazonS3Client])

(defn s3-client?
  [x]
  (instance? AmazonS3Client x))

(s/def ::channel any?) ;; for docs
(s/def ::store any?) ;; for docs
(s/def ::s3-client s3-client?)
(s/def ::non-empty-string (s/and string? #(not-empty %)))
(s/def ::path-elem (s/and ::non-empty-string
                     #(not (str/index-of % "/"))))
(s/def ::key-name ::path-elem)
(s/def ::keyfile-name ::non-empty-string)
(s/def ::parsed-access-key-id (s/keys :req-un [::key-name ::keyfile-name]))
(s/def ::secret ::non-empty-string)
(s/def ::created inst?)
(s/def ::creds (s/keys :req-un [::key-name ::secret ::created]))
(s/def ::current ::creds)
(s/def ::previous ::creds)
(s/def ::access-key-id ::non-empty-string)
(s/def ::keyfile (s/keys :req-un [::current]
                         :opt-un [::previous]))

(s/def ::creds-result-success (s/keys :req-un [::creds ::access-key-id]))
(s/def ::creds-result (s/nilable ::creds-result-success))

(in-ns 'cognitect.s3-creds.store)
(require '[clojure.spec.alpha :as s])
(alias 's3-creds 'cognitect.s3-creds)
(alias 'anom 'cognitect.anomalies)
(s/def ::val any?)
(s/def ::val-return (s/keys :req-un [::val]))
(s/fdef get-val
        :args (s/or :client (s/cat :s3-client ::s3-client
                                   :access-key-id ::s3-creds/access-key-id)
                    :no-client (s/cat :access-key-id ::s3-creds/access-key-id))
        :ret (s/or :success ::val-return
                   :fail ::anom/anomaly))
(s/fdef put-val
        :args (s/or :client (s/cat :s3-client ::s3-client
                                   :access-key-id ::s3-creds/access-key-id
                                   :v ::val)
                    :no-client  (s/cat :access-key-id ::s3-creds/access-key-id
                                       :v ::val))
        :ret (s/or :success ::val-return
                   :fail ::anom/anomaly))

(in-ns 'cognitect.s3-creds.keyfile)
(require '[clojure.spec.alpha :as s])
(alias 's3-creds 'cognitect.s3-creds)
(s/fdef parse-access-key-id
        :args (s/cat :s string?)
        :ret (s/nilable ::s3-creds/parsed-access-key-id))
(s/fdef create-access-key-id
        :args (s/cat :keyfile-name ::s3-creds/keyfile-name :key-name ::s3-creds/key-name)
        :ret (s/nilable string?))
(s/fdef create-creds
        :args (s/cat)
        :ret ::s3-creds/creds)
(s/fdef rotate-creds
        :args (s/? (s/nilable ::s3-creds/keyfile))
        :ret ::s3-creds/keyfile)
(s/fdef matching-creds
        :args (s/cat :creds (s/nilable ::s3-creds/creds) :key-name ::s3-creds/key-name)
        :ret (s/nilable ::s3-creds/creds))
(s/fdef creds-for-key-name
        :args (s/cat :keyfile ::s3-creds/keyfile :key-name ::s3-creds/key-name)
        :ret (s/nilable ::s3-creds/creds))
(s/fdef creds
        :args (s/cat :pakid ::s3-creds/parsed-access-key-id :keyfile ::s3-creds/keyfile)
        :ret ::s3-creds/creds-result)
